import { EventEmitter, FetchError, Storage } from '@aracna/core';
import { ConnectionOptions, TLSSocket } from 'tls';
import { FcmClientACG, FcmClientData, FcmClientECE, FcmClientEvents, FcmClientInit, FcmClientOptions } from '../definitions/interfaces.js';
/**
 * The FcmClient class establishes a TLS socket connection with Google MTalk and handles the communication.
 *
 * - The client needs a valid ACG ID and ACG security token to login, they can be obtained with the `registerToFCM` function.
 * - The client needs a valid ECE auth secret and ECE private key to decrypt the messages, they can be obtained with the `createFcmECDH` and `generateFcmSalt` functions.
 * - The client will store the received persistent IDs in the storage. By default the storage is an in-memory storage, it is reccomended to use a persistent storage.
 * - The client will perform a heartbeat every 5 seconds to avoid losing the socket connection.
 *
 * Events:
 *
 * - The client will emit a `close` event when a close message is received.
 * - The client will emit a `heartbeat` event when a heartbeat is received.
 * - The client will emit a `iq` event when an IQ stanza is received.
 * - The client will emit a `login` event when a login response is received.
 * - The client will emit a `message` event when a message is received.
 * - The client will emit a `message-data` event when a message is decrypted.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/fcm/classes/fcm-client)
 */
export declare class FcmClient extends EventEmitter<FcmClientEvents> {
    /**
     * The ACG object, contains the ACG ID and ACG security token.
     */
    protected readonly acg: FcmClientACG;
    /**
     * The data object, contains all things relative to the exchange of messages.
     */
    protected data: FcmClientData;
    /**
     * The ECE object, contains the auth secret and the ECDH private key.
     */
    protected readonly ece: FcmClientECE;
    /**
     * The options object, contains the heartbeat frequency.
     */
    protected readonly options: FcmClientOptions;
    /**
     * The TLS socket.
     */
    protected socket?: TLSSocket;
    /**
     * The storage instance.
     */
    protected readonly storage: Storage;
    /**
     * The storage key.
     */
    protected readonly storageKey: string;
    constructor(init?: FcmClientInit);
    /**
     * Connects the socket to Google MTalk.
     *
     * - The persistent IDs received in the previous session will be retrieved from the storage to communicate them to the server during the login.
     * - The ACG ID and ACG security token will be verified before connecting.
     */
    connect(options?: ConnectionOptions): Promise<void | FetchError | Error>;
    /**
     * Disconnects the socket.
     */
    disconnect(error?: Error): Promise<void>;
    /**
     * Sends a login request message to the socket.
     */
    protected login(): void;
    /**
     * Sends a heartbeat ping message to the socket.
     */
    protected heartbeat(): void;
    /**
     * Prepares the data for the next message.
     */
    protected prepareForNextMessage(): void;
    protected onSocketClose: (error: boolean) => void;
    protected onSocketConnect: () => void;
    /**
     * Handles the data received from the socket.
     *
     * - The data is always concatenated to the previous data.
     * - The data is handled based on the current state.
     *
     * States:
     *
     * - VERSION_TAG_AND_SIZE: Contains an extra byte at the beginning with the MCS version.
     * - TAG_AND_SIZE: Contains a tag which is always 1 byte and tells the type of the message, and a size which is variable in length and tells the size of the message.
     * - SIZE: Contains the size of the message.
     * - BYTES: Contains the message.
     */
    protected onSocketData: (data: Buffer) => void;
    /**
     * Reads the MCS version from first byte of the data.
     *
     * - Moves the cursor 1 byte forward.
     * - Sets the state to TAG_AND_SIZE.
     */
    protected onSocketDataVersion: () => void;
    /**
     * Reads the tag from the first byte of the data, or the second byte if the version was read before.
     *
     * - Moves the cursor 1 byte forward.
     * - Sets the state to SIZE.
     */
    protected onSocketDataTag: () => void;
    /**
     * Reads the size from the data, the size can range from 1 to 5 bytes.
     *
     * - Aborts and waits for more bytes if there are not enough bytes to read.
     * - Aborts and prepares for the next message if the message was not decodable or decryptable with the maximum amount of bytes allowed for the size packet.
     * - Tries to decode the message with the current size and bytes, if it fails it increases the length of the size packet by 1 and tries again.
     * - Once the message is decodable and decryptable it moves the cursor by the size of the size packet and sets the state to BYTES.
     */
    protected onSocketDataSize: () => void;
    protected onSocketDataSizeTimeout: () => void;
    /**
     * Decodes the message based on the tag and prepares for the next message.
     *
     * Close:
     *
     * - Emits the `close` event.
     *
     * DataMessageStanza:
     *
     * - Emits the `message` event.
     * - Pushes the persistent ID of the message to the received persistent IDs and stores them.
     * - Decrypts the message data.
     * - Emits the `message-data` event with the decrypted data.
     *
     * HeartbeatAck:
     *
     * - Emits the `heartbeat` event.
     * - Schedules the next heartbeat.
     *
     * IqStanza:
     *
     * - Emits the `iq` event.
     *
     * LoginResponse:
     *
     * - Emits the `login` event.
     * - Resets the received persistent IDs and stores them.
     * - Calls the `heartbeat` function.
     */
    protected onSocketDataBytes: () => void;
    protected onSocketDrain: () => void;
    protected onSocketEnd: () => void;
    protected onSocketError: (error: Error) => void;
    protected onSocketKeylog: (line: Buffer) => void;
    protected onSocketLookup: (error: Error, address: string, family: string, host: string) => void;
    protected onSocketOCSPResponse: (response: Buffer) => void;
    /**
     * Calls the `login` function when the socket is ready.
     */
    protected onSocketReady: () => void;
    protected onSocketSecureConnect: () => void;
    protected onSocketSession: (session: Buffer) => void;
    protected onSocketTimeout: () => void;
    /**
     * Returns the ACG ID.
     */
    getAcgID(): bigint;
    /**
     * Returns the ACG security token.
     */
    getAcgSecurityToken(): bigint;
    /**
     * Returns the auth secret.
     */
    getAuthSecret(): ArrayLike<number>;
    /**
     * Returns the ECDH private key.
     */
    getEcdhPrivateKey(): ArrayLike<number>;
    /**
     * Returns the heartbeat frequency.
     */
    getHeartbeatFrequency(): number;
    /**
     * Returns the TLS socket.
     */
    getSocket(): TLSSocket | undefined;
    /**
     * Returns the storage.
     */
    getStorage(): Storage;
    /**
     * Returns the storage key.
     */
    getStorageKey(): string;
    /**
     * Sets the ACG ID.
     */
    setAcgID(id: bigint): this;
    /**
     * Sets the ACG security token.
     */
    setAcgSecurityToken(securityToken: bigint): this;
    /**
     * Sets the auth secret.
     */
    setAuthSecret(authSecret: ArrayLike<number>): this;
    /**
     * Sets the ECDH private key.
     */
    setEcdhPrivateKey(privateKey: ArrayLike<number>): this;
    /**
     * Sets the heartbeat frequency in ms.
     */
    setHeartbeatFrequency(ms: number): this;
}
