import { CookieParseOptions } from 'cookie';
import { CookieSerializeOptions } from 'cookie';

export declare const ALPHABET_ALPHANUMERIC: string;

export declare const ALPHABET_HEX_LOWERCASE: string;

export declare const ALPHABET_HEX_UPPERCASE: string;

/** */
export declare const ALPHABET_LOWERCASE: string;

export declare const ALPHABET_NO_LOOK_ALIKES: string;

export declare const ALPHABET_NO_LOOK_ALIKES_SAFE: string;

export declare const ALPHABET_NUMBERS: string;

export declare const ALPHABET_UPPERCASE: string;

export declare enum AnsiColor {
    BLACK = "\u001B[30m",
    RED = "\u001B[31m",
    GREEN = "\u001B[32m",
    YELLOW = "\u001B[33m",
    BLUE = "\u001B[34m",
    MAGENTA = "\u001B[35m",
    CYAN = "\u001B[36m",
    WHITE = "\u001B[37m",
    RESET = "\u001B[0m"
}

/**
 * The Appearence class manages the theme of anything that can have an appearence.
 *
 * - The theme will persist to a storage of your choice, by default it will be stored in memory.
 * - The default theme is system, which means that the theme will be dark or light depending on the system theme.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/appearence)
 */
export declare class Appearence extends EventEmitter<AppearenceEvents> {
    /**
     * The storage that will be used to store the theme.
     */
    storage: Storage_2;
    /**
     * The key that will be used to store the theme.
     */
    storageKey: string;
    /**
     * The theme, can be dark, light or system.
     */
    theme: Theme;
    constructor(init?: AppearenceInit);
    /**
     * Retrieves the theme from the storage and sets it.
     */
    initialize(): Promise<boolean>;
    /**
     * Toggles the theme between dark and light.
     * If the theme is set to system, it will be set to dark or light depending on the system theme.
     */
    toggleTheme(): this;
    /**
     * Sets the theme.
     * The "change-theme" event will be emitted.
     */
    setTheme(theme: Theme): this;
    /**
     * Stores the theme in the storage.
     */
    store(): Promise<boolean>;
    /**
     * Registers the theme event listener in environments that support it.
     */
    protected registerThemeEventListener(): void;
    /**
     * Returns the theme depending on the system theme.
     */
    get themeByPrefersColorScheme(): Theme;
    /**
     * Checks if the theme is dark.
     */
    get isThemeDark(): boolean;
    /**
     * Checks if the theme is light.
     */
    get isThemeLight(): boolean;
    /**
     * Checks if the theme is system.
     */
    get isThemeSystem(): boolean;
}

export declare interface AppearenceEvents extends EventEmitterEvents {
    'change-theme': (theme: Theme) => any;
}

export declare interface AppearenceInit {
    storage?: {
        instance?: Storage_2;
        key?: string;
    };
    theme?: Theme;
}

/**
 * Appends the search params to a URL.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/url)
 */
export declare function appendSearchParamsToURL<T extends URLSearchParamsRecord>(url: string, params: AppendSearchParamsToURLParams<T>, base?: string | URL): string;

export declare function appendSearchParamsToURL<T extends URLSearchParamsRecord>(url: URL, params: AppendSearchParamsToURLParams<T>, base?: string | URL): URL;

export declare type AppendSearchParamsToURLParams<T extends URLSearchParamsRecord = URLSearchParamsRecord> = DeserializeURLSearchParamsInit<T>;

/**
 * The AracnaBlob class is built on top of the Blob class.
 *
 * - The data contained in the Blob can be resolved asynchronously and accessed at a later time from the instance itself.
 * - The instance supports JSON serialization and deserialization out of the box unlike the Blob class.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/aracna-blob)
 */
export declare class AracnaBlob {
    private _arrayBuffer?;
    /**
     * The Blob instance.
     */
    readonly blob: Blob;
    /**
     * The unique identifier of the instance.
     */
    readonly id: string;
    private _text?;
    constructor(blob: Blob);
    constructor(json: AracnaBlobJSON);
    /**
     * Resolves the data contained in the Blob as an ArrayBuffer.
     */
    resolveArrayBuffer(): Promise<void>;
    /**
     * Resolves the data contained in the Blob as a string.
     */
    resolveText(): Promise<void>;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Blob/slice) */
    slice(start?: number, end?: number, contentType?: string): Blob;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Blob/stream) */
    stream(): ReadableStream<Uint8Array>;
    stream(): NodeJS.ReadableStream;
    /**
     * Serializes the instance into a JSON object.
     */
    toJSON(): AracnaBlobJSON;
    /**
     * Returns the data contained in the Blob as an ArrayBuffer.
     * You need to call the "resolveArrayBuffer" method before accessing this property.
     */
    get arrayBuffer(): ArrayBuffer;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Blob/size) */
    get size(): number;
    /**
     * Returns the data contained in the Blob as a string.
     * You need to call the "resolveText" method before accessing this property.
     */
    get text(): string;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Blob/type) */
    get type(): string;
    /**
     * Returns the data contained in the Blob as a Uint8Array.
     * You need to call the "resolveArrayBuffer" or "resolveText" method before accessing this property.
     */
    get uInt8Array(): Uint8Array;
    /**
     * Returns an empty AracnaBlob instance.
     */
    static get EMPTY(): AracnaBlob;
}

export declare interface AracnaBlobJSON {
    id: string;
    size: number;
    type: string;
    uInt8Array: Uint8Array;
}

/**
 * The AracnaFile class extends the AracnaBlob class and is built on top of the File class.
 *
 * - The data contained in the File can be resolved asynchronously and accessed at a later time from the instance itself.
 * - The instance supports JSON serialization and deserialization out of the box unlike the File class.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/aracna-file)
 */
export declare class AracnaFile extends AracnaBlob {
    /**
     * The File instance.
     */
    readonly file: File;
    constructor(file: File);
    constructor(json: AracnaFileJSON);
    toJSON(): AracnaFileJSON;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/File/lastModified) */
    get lastModified(): number;
    /**
     * Returns the last modified date of the file.
     */
    get lastModifiedDate(): Date;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/File/name) */
    get name(): string;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/File/webkitRelativePath) */
    get webkitRelativePath(): string;
    /**
     * Returns an empty AracnaFile instance.
     */
    static get EMPTY(): AracnaFile;
}

export declare interface AracnaFileJSON extends AracnaBlobJSON {
    lastModified: number;
    name: string;
    webkitRelativePath: string;
}

/**
 * The AsyncStorage class is an abstraction to implement any asynchronous storage API in an uniform way.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/async-storage)
 */
export declare class AsyncStorage extends Storage_3 {
    constructor(name: string, clear: () => Promise<void>, get: <T extends StorageItem>(key: string) => Promise<T>, has: (key: string) => Promise<boolean>, remove: (key: string) => Promise<void>, set: <T extends StorageItem>(key: string, item: T) => Promise<void>);
    /**
     * Clears the storage, removing all the items.
     */
    clear(): Promise<void | Error>;
    /**
     * Retrieves an item from the storage.
     */
    get<T extends StorageItem>(key: string): Promise<T | Error>;
    /**
     * Removes an item from the storage.
     * Optionally you can specify the keys of the item that you want to remove, if you don't specify any key the whole item will be removed.
     */
    remove<T extends StorageItem>(key: string, keys?: KeyOf.Deep<T>[]): Promise<void | Error>;
    /**
     * Sets an item in the storage.
     * Optionally you can specify the keys of the item that you want to set, if you don't specify any key the whole item will be set.
     */
    set<T extends StorageItem>(key: string, item: T, keys?: KeyOf.Deep<T>[]): Promise<void | Error>;
    /**
     * Copies an item from the storage to a target object.
     * Optionally you can specify the keys of the item that you want to copy, if you don't specify any key the whole item will be copied.
     */
    copy<T1 extends StorageItem, T2 extends StorageTarget = StorageTarget, T extends T1 & T2 = T1 & T2>(key: string, target: T2, keys?: KeyOf.Deep<T>[]): Promise<void | Error>;
    /**
     * Checks if an item exists in the storage.
     * Optionally you can specify the keys of the item that you want to check, if you don't specify any key the whole item will be checked.
     */
    has<T extends StorageItem>(key: string, keys?: KeyOf.Deep<T>[]): Promise<boolean>;
}

/**
 * The `cafs` function stands for `call async functions sequentially`. It executes a list of async functions in sequence.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/cafs)
 */
export declare function cafs(...fns: ((...args: any[]) => Promise<any>)[]): Promise<void>;

/**
 * The `cafsue` function stands for `call async functions sequentially until error`. It executes a list of async functions in sequence, the execution stops if a function throws or returns an error.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/cafsue)
 */
export declare function cafsue(...fns: ((...args: any[]) => Promise<any>)[]): Promise<void>;

/**
 * The `cafsueof` function stands for `call async functions sequentially until error or falsy`. It executes a list of async functions in sequence, the execution stops if a function throws or returns an error or a falsy value.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/cafsueof)
 */
export declare function cafsueof(...fns: ((...args: any[]) => Promise<any>)[]): Promise<boolean>;

/**
 * @deprecated
 */
export declare function chainPromises(...fns: ((...args: any[]) => Promise<any>)[]): Promise<void>;

/**
 * @deprecated
 */
export declare function chainTruthyPromises(...fns: ((...args: any[]) => Promise<any>)[]): Promise<boolean>;

declare type Clear = () => ClearReturn;

/**
 * Clears all intervals.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/interval)
 */
export declare function clearEveryInterval(): void;

/**
 * Clears all timeouts.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/timeout)
 */
export declare function clearEveryTimeout(): void;

/**
 * Clears an interval.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/interval)
 */
declare function clearInterval_2(key: IntervalMapKey): void;
export { clearInterval_2 as clearInterval }

declare type ClearReturn = void | Error | Promise<void | Error>;

/**
 * Clears a timeout.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/timeout)
 */
declare function clearTimeout_2(key: TimeoutMapKey): void;
export { clearTimeout_2 as clearTimeout }

/**
 * Creates a copy of an array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/array)
 */
export declare function cloneArray<T>(array: T[]): T[];

/**
 * Creates a copy of an object.
 * Optionally the copy can be deep.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function cloneObject<T extends object>(object: T, options?: CloneObjectOptions): T;

export declare interface CloneObjectOptions {
    deep: boolean;
}

/**
 * Concatenates an URL with pathnames.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/url)
 */
export declare function concatURL(url: string, ...pathnames: Partial<string>[]): string;

export declare function concatURL(url: URL, ...pathnames: Partial<string>[]): URL;

/**
 * The Cookie class is an abstraction to implement any cookie API in an uniform way.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/cookie)
 */
export declare class Cookie {
    /**
     * The cookie instance name.
     */
    readonly name: string;
    /**
     * The cookie separator used to separate the cookie name from the cookie item key.
     */
    readonly separator: string;
    private readonly _get;
    private readonly _set;
    constructor(name: string, get: () => string, set: (cookies: string) => void, separator?: string);
    /**
     * Clears all cookies.
     */
    clear(options?: CookieSerializeOptions): void | Error;
    /**
     * Retrieves an item from the cookies.
     */
    get<T extends CookieItem>(key: string): T | Error;
    /**
     * Removes an item from the cookies.
     * Optionally you can specify the keys of the item that you want to remove, if you don't specify any key the whole item will be removed.
     */
    remove<T extends CookieItem>(key: string, options?: CookieSerializeOptions, keys?: KeyOf.Shallow<T>[]): void | Error;
    /**
     * Sets an item in the cookies.
     * Optionally you can specify the keys of the item that you want to set, if you don't specify any key the whole item will be set.
     */
    set<T extends CookieItem>(key: string, item: T, options?: CookieSerializeOptions, keys?: KeyOf.Shallow<T>[]): void | Error;
    /**
     * Copies an item from the cookies to a target object.
     * Optionally you can specify the keys of the item that you want to copy, if you don't specify any key the whole item will be copied.
     */
    copy<T1 extends CookieItem, T2 extends CookieItem = CookieItem, T extends T1 & T2 = T1 & T2>(key: string, target: T2, keys?: KeyOf.Shallow<T>[]): void | Error;
    /**
     * Checks if an item exists in the cookies.
     * Optionally you can specify the keys of the item that you want to check, if you don't specify any key the whole item will be checked.
     */
    has<T extends CookieItem>(key: string, keys?: KeyOf.Shallow<T>[]): boolean;
    protected deserialize(options?: CookieParseOptions): CookieObject | Error;
    protected serialize(key: string, value: string, options?: CookieSerializeOptions): string | Error;
    protected serialize<T extends CookieItem>(key: string, ik: keyof T, value: Primitive, options?: CookieSerializeOptions): string | Error;
    protected toCookieItemKey<T extends CookieItem>(key: string, ik: KeyOf.Shallow<T>): string;
    protected toDocumentCookieName<T extends CookieItem>(key: string, ik: KeyOf.Shallow<T>): string;
}

export declare interface CookieItem extends Record<PropertyKey, Primitive> {
}

export declare interface CookieObject extends Record<string, string> {
}

export declare interface CookieSource {
    get: () => string;
    set: (string: string) => void;
}

/**
 * Copies a property from one object to another. The key supports bracket and dot notation.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function copyObjectProperty<T1 extends object, T2 extends object, T extends T1 & T2>(source: T, key: KeyOf.Deep<T>, target: T): void | Error;

export declare function copyObjectProperty<T1 extends object, T2 extends object, T extends T1 & T2>(source: T, key: string, target: T): void | Error;

declare type CopyReturn = void | Error | Promise<void | Error>;

export declare const CoreClassLogger: Logger;

/**
 * The Configuration class is used to configure the library.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/configuration)
 */
export declare class CoreConfiguration {
    /**
     * The functions configuration.
     */
    static functions: CoreConfigurationFunctions;
}

export declare interface CoreConfigurationFunctions {
    tc: {
        log: boolean;
        onCatch: <T extends Error>(error: T, log: boolean) => any;
    };
    tcp: {
        log: boolean;
        onCatch: <T extends Error>(error: T, log: boolean) => any;
    };
}

export declare const CoreFunctionLogger: Logger;

export declare enum CoreLoggerName {
    CLASS = "CORE_CLASS",
    FUNCTION = "CORE_FUNCTION",
    UTIL = "CORE_UTIL"
}

export declare enum CoreStorageName {
    MEMORY = "CORE_MEMORY"
}

export declare const CoreUtilLogger: Logger;

/**
 * Counts the number of headers in a `FetchRequestInit` or `RequestInit` object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function countFetchRequestInitHeaders<T>(init: FetchRequestInit<T> | RequestInit): number;

/**
 * The `debounce` function is used to prevent a function from being called too many times in a short period.
 * The function will only be called after it stops being called for the specified amount of time.
 *
 * Optionally the key can be specified, otherwise the function itself will be used as the key.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/debounce)
 */
export declare function debounce(fn: Function, ms: number, key?: DebounceMapKey): void;

export declare type DebounceMapKey = bigint | number | string | symbol | Function;

export declare type DebounceMapValue = NodeJS.Timeout | number;

/**
 * Decodes a base16 string into a Uint8Array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/base16)
 */
export declare function decodeBase16(string: string, options?: DecodeBase16Options): Uint8Array;

export declare interface DecodeBase16Options {
    loose?: boolean;
    out?: new (size: number) => {
        [index: number]: number;
    };
}

/**
 * Decodes a base32 string into a Uint8Array.
 *
 * [Araña Reference](https://arana.dariosechi.it/core/utils/base32)
 */
export declare function decodeBase32(string: string, options?: DecodeBase32Options): Uint8Array;

/**
 * Decodes a base32hex string into a Uint8Array.
 *
 * [Araña Reference](https://arana.dariosechi.it/core/utils/base32)
 */
export declare function decodeBase32Hex(string: string, options?: DecodeBase32HexOptions): Uint8Array;

export declare interface DecodeBase32HexOptions extends DecodeBase32Options {
}

export declare interface DecodeBase32Options extends DecodeBase16Options {
}

/**
 * Decodes a base64 string into a Uint8Array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/base64)
 */
export declare function decodeBase64(string: string, options?: DecodeBase64Options): Uint8Array;

export declare interface DecodeBase64Options extends DecodeBase16Options {
}

/**
 * Decodes a base64url string into a Uint8Array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/base64)
 */
export declare function decodeBase64URL(string: string, options?: DecodeBase64URLOptions): Uint8Array;

export declare interface DecodeBase64URLOptions extends DecodeBase64Options {
}

export declare function decodeJSON<T extends object>(text: string, options?: DecodeJsonOptions): T | Error;

export declare function decodeJSON<T extends object>(text: string, options: DecodeJsonOptions | undefined, fallback: T): T;

export declare interface DecodeJsonOptions {
    castBigIntStringToBigInt?: boolean;
    castFloatStringToNumber?: boolean;
    castIntStringToNumber?: boolean;
    castUnsafeIntToBigInt?: boolean;
}

/**
 * Decodes a buffer into a string.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/text)
 */
export declare function decodeText(input: AllowSharedBufferSource, options?: TextDecodeOptions): string;

/** */
export declare const DEFAULT_APPEARENCE_STORAGE_KEY: string;

/** */
export declare const DEFAULT_LOCALIZATION_STORAGE_KEY: string;

/** */
export declare const DEFAULT_QUEUE_CONCURRENCY: number;

export declare const DEFAULT_QUEUE_DELAY: number;

export declare const DEFAULT_QUEUE_TIMEOUT: number;

/** */
export declare const DEFAULT_TYPEAHEAD_DEBOUNCE_TIME: number;

/** */
export declare const DEFAULT_WF_MS: number;

export declare const DEFAULT_WF_TIMEOUT: number;

export declare const DEFAULT_WFP_MS: number;

export declare const DEFAULT_WFP_TIMEOUT: number;

/**
 * The DeferredPromise class is built on top of the native Promise class. It provides a way to resolve or reject a promise from the outside.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/deferred-promise)
 */
export declare class DeferredPromise<T> {
    /**
     * The native promise instance.
     */
    readonly instance: Promise<T>;
    /**
     * The reason for the rejection of the promise.
     */
    reason?: any;
    /**
     * The state of the promise, can be pending, fulfilled or rejected.
     */
    state: PromiseState;
    /**
     * The value of the resolved promise.
     */
    value?: T;
    private _reject;
    private _resolve;
    constructor();
    reject: (reason?: any) => void;
    /**
     * Resolves the promise with a value or the result of another promise.
     */
    resolve: (value: T | PromiseLike<T>) => void;
    /**
     * Attaches a callback for only the rejection of the Promise.
     * @param onrejected The callback to execute when the Promise is rejected.
     * @returns A Promise for the completion of the callback.
     */
    catch<TResult = never>(onrejected?: ((reason: any) => TResult | PromiseLike<TResult>) | undefined | null): this;
    /**
     * Attaches a callback that is invoked when the Promise is settled (fulfilled or rejected). The
     * resolved value cannot be modified from the callback.
     * @param onfinally The callback to execute when the Promise is settled (fulfilled or rejected).
     * @returns A Promise for the completion of the callback.
     */
    finally(onfinally?: (() => void) | undefined | null): this;
    /**
     * Attaches callbacks for the resolution and/or rejection of the Promise.
     * @param onfulfilled The callback to execute when the Promise is resolved.
     * @param onrejected The callback to execute when the Promise is rejected.
     * @returns A Promise for the completion of which ever callback is executed.
     */
    then<TResult1 = T, TResult2 = never>(onfulfilled?: ((value: T) => TResult1 | PromiseLike<TResult1>) | undefined | null, onrejected?: ((reason: any) => TResult2 | PromiseLike<TResult2>) | undefined | null): this;
    /**
     * Checks if the promise is fulfilled.
     */
    get isFulfilled(): boolean;
    /**
     * Checks if the promise is pending.
     */
    get isPending(): boolean;
    /**
     * Checks if the promise is rejected.
     */
    get isRejected(): boolean;
    /**
     * Checks if the promise is resolved.
     */
    get isResolved(): boolean;
}

/**
 * Deletes a header from a `FetchRequestInit` or `RequestInit` object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function deleteFetchRequestInitHeader<T>(init: FetchRequestInit<T> | RequestInit, name: string): void;

/**
 * Deletes the properties of an object that either match the predicate or are in the list of keys. The keys support bracket and dot notation.
 * Optionally deletes deep properties as well.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function deleteObjectProperties<T extends object>(object: T, predicate: DeleteObjectPropertiesPredicate, options: DeleteObjectPropertiesOptions & {
    deep: true;
}): void;

export declare function deleteObjectProperties<T extends object>(object: T, predicate: DeleteObjectPropertiesPredicate<T>, options?: DeleteObjectPropertiesOptions): void;

export declare function deleteObjectProperties<T extends object>(object: T, keys: KeyOf.Deep<T>[]): void;

export declare function deleteObjectProperties<T extends object>(object: T, keys: KeyOf.Shallow<T>[]): void;

export declare function deleteObjectProperties<T extends object>(object: T, keys: string[]): void;

export declare interface DeleteObjectPropertiesOptions {
    deep: boolean;
}

export declare type DeleteObjectPropertiesPredicate<T extends object = object, K extends keyof T | string = keyof T | string> = (object: T, key: K, value: T extends Record<K, infer V> ? V : unknown, keys?: PropertyKey[]) => boolean;

/**
 * Deletes a property from an object. The key supports bracket and dot notation.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function deleteObjectProperty<T extends object>(object: T, key: KeyOf.Deep<T>): void;

export declare function deleteObjectProperty<T extends object>(object: T, key: string): void;

/**
 * Deserializes a `Blob` object into an `AracnaBlob` instance.
 * Optionally resolves the data of the blob.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/blob)
 */
export declare function deserializeBlob(blob: Blob, options?: DeserializeBlobOptions): Promise<AracnaBlob>;

export declare interface DeserializeBlobOptions {
    resolveArrayBuffer?: boolean;
    resolveText?: boolean;
}

/**
 * Deserializes a cookie string to an object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/cookie)
 */
export declare function deserializeCookie(cookie: string, options?: CookieParseOptions): CookieObject;

/**
 * Deserializes a `File` object into an `AracnaFile` instance.
 * Optionally resolves the data of the file.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/file)
 */
export declare function deserializeFile(file: File, options?: DeserializeFileOptions): Promise<AracnaFile>;

export declare interface DeserializeFileOptions extends DeserializeBlobOptions {
}

/**
 * Deserializes a `FormData` object into a plain object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/form-data)
 */
export declare function deserializeFormData<T extends object>(data: FormData, options?: DeserializeFormDataOptions): T;

declare interface DeserializeFormDataOptions {
    json?: DecodeJsonOptions;
}

/**
 * Deserializes an array, string, plain object or `URLSearchParams` to an array, string or plain object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/url)
 */
export declare function deserializeURLSearchParams<T extends URLSearchParamsRecord>(params: DeserializeURLSearchParamsInit): T;

export declare function deserializeURLSearchParams<T extends URLSearchParamsRecord>(params: DeserializeURLSearchParamsInit, type: 'string'): string;

export declare function deserializeURLSearchParams<T extends URLSearchParamsRecord>(params: DeserializeURLSearchParamsInit, type: 'array'): string[][];

export declare function deserializeURLSearchParams<T extends URLSearchParamsRecord>(params: DeserializeURLSearchParamsInit, type: 'object'): T;

export declare type DeserializeURLSearchParamsInit<T extends URLSearchParamsRecord = URLSearchParamsRecord> = string | string[][] | T | URLSearchParams;

export declare type DeserializeURLSearchParamsType = 'array' | 'object' | 'string';

/** */
export declare const EMPTY_OBJECT: () => Record<PropertyKey, any>;

/**
 * Encodes an array of integers into a base16 string.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/base16)
 */
export declare function encodeBase16(array: ArrayLike<number>, options?: EncodeBase16Options): string;

export declare interface EncodeBase16Options {
    pad?: boolean;
}

/**
 * Encodes an array of integers into a base32 string.
 *
 * [Araña Reference](https://arana.dariosechi.it/core/utils/base32)
 */
export declare function encodeBase32(array: ArrayLike<number>, options?: EncodeBase32Options): string;

/**
 * Encodes an array of integers into a base32hex string.
 *
 * [Araña Reference](https://arana.dariosechi.it/core/utils/base32)
 */
export declare function encodeBase32Hex(array: ArrayLike<number>, options?: EncodeBase32HexOptions): string;

export declare interface EncodeBase32HexOptions extends EncodeBase32Options {
}

export declare interface EncodeBase32Options extends EncodeBase16Options {
}

/**
 * Encodes an array of integers into a base64 string.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/base64)
 */
export declare function encodeBase64(array: ArrayLike<number>, options?: EncodeBase64Options): string;

export declare interface EncodeBase64Options extends EncodeBase16Options {
}

/**
 * Encodes an array of integers into a base64url string.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/base64)
 */
export declare function encodeBase64URL(array: ArrayLike<number>, options?: EncodeBase64URLOptions): string;

export declare interface EncodeBase64URLOptions extends EncodeBase64Options {
}

export declare function encodeJSON(value: unknown, options?: EncodeJsonOptions): string | Error;

export declare function encodeJSON(value: unknown, options: EncodeJsonOptions | undefined, fallback: string): string;

export declare interface EncodeJsonOptions<T extends JsonEncoding = 'utf-8'> {
    castBigIntToString?: boolean;
    encoding?: T;
}

/**
 * Encodes a string into a Uint8Array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/text)
 */
export declare function encodeText(input: string): Uint8Array;

/**
 * @deprecated
 */
export declare class Environment {
    static get(key: string): string | undefined;
    static has(key: string): boolean;
    static get isBlobDefined(): boolean;
    static get isBlobNotDefined(): boolean;
    static get isDevelopment(): boolean;
    static get isNotDevelopment(): boolean;
    static get isDocumentDefined(): boolean;
    static get isDocumentNotDefined(): boolean;
    static get isFetchDefined(): boolean;
    static get isFetchNotDefined(): boolean;
    static get isFileDefined(): boolean;
    static get isFileNotDefined(): boolean;
    static get isFormDataDefined(): boolean;
    static get isFormDataNotDefined(): boolean;
    static get isJestDefined(): boolean;
    static get isJestNotDefined(): boolean;
    static get isProduction(): boolean;
    static get isNotProduction(): boolean;
    static get isTest(): boolean;
    static get isNotTest(): boolean;
    static get isProcessDefined(): boolean;
    static get isProcessNotDefined(): boolean;
    static get isWindowDefined(): boolean;
    static get isWindowNotDefined(): boolean;
    static get NODE_ENV(): string | undefined;
}

/**
 * The EventEmitter class is used to emit events and register listeners.
 * The API is based on the Node.js EventEmitter API.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/event-emitter)
 */
export declare class EventEmitter<T extends EventEmitterEvents = EventEmitterEvents> {
    protected listeners: EventEmitterListener<T, keyof T>[];
    protected maxListeners: number;
    constructor();
    /**
     * Counts the listeners that match the name, callback and options.
     * If no arguments are passed, all listeners will be counted.
     */
    countListeners<K extends keyof T>(name?: K, callback?: T[K], options?: EventEmitterListenerOptions): number;
    /**
     * Emits an event.
     */
    emit<K extends keyof T>(name: K, ...args: Parameters<T[K]>): boolean;
    /**
     * Returns the names of the events that have listeners.
     */
    getEventNames(): (keyof T)[];
    /**
     * Returns the listeners that match the name, callback and options.
     * If no arguments are passed, all listeners will be returned.
     */
    getListeners<K extends keyof T>(name?: K, callback?: T[K], options?: EventEmitterListenerOptions): EventEmitterListener<T, keyof T>[];
    /**
     * Returns the maximum number of listeners that can be registered for a single event.
     */
    getMaxListeners(): number;
    /**
     * Checks if the event has listeners that match the name, callback and options.
     */
    hasListeners<K extends keyof T>(name?: K, callback?: T[K], options?: EventEmitterListenerOptions): boolean;
    /**
     * Removes the listeners that match the name, callback and options.
     * If no arguments are passed, all listeners will be removed.
     */
    off<K extends keyof T>(name?: K, callback?: T[K], options?: EventEmitterListenerOptions): this;
    /**
     * Adds a listener, which will be called when the event is emitted.
     *
     * Optionally the listener can be removed after the first call with the `once` option.
     * Optionally the listener can be prepended to the listeners array with the `prepend` option.
     */
    on<K extends keyof T>(name: K, callback: T[K], options?: EventEmitterListenerOptions): this;
    /**
     * Adds a listener, which will be called only once when the event is emitted.
     */
    once<K extends keyof T>(name: K, callback: T[K], options?: EventEmitterListenerOptions): this;
    /**
     * Adds a listener, which will be called first when the event is emitted.
     */
    prepend<K extends keyof T>(name: K, callback: T[K], options?: EventEmitterListenerOptions): this;
    /**
     * Sets the listeners.
     */
    setListeners(listeners: EventEmitterListener<T, keyof T>[]): this;
    /**
     * Sets the maximum number of listeners that can be registered for a single event.
     */
    setMaxListeners(n: number): this;
}

export declare type EventEmitterEvents = Record<EventEmitterListenerName, EventEmitterListenerCallback>;

export declare interface EventEmitterListener<T extends EventEmitterEvents = EventEmitterEvents, K extends keyof T = keyof T> {
    callback: T[K];
    name: K;
    options?: EventEmitterListenerOptions;
}

export declare type EventEmitterListenerCallback = (...args: any[]) => any;

export declare type EventEmitterListenerName = string | symbol;

export declare interface EventEmitterListenerOptions {
    once?: boolean;
    prepend?: boolean;
}

/**
 * The Fetch class is built on top of the native Fetch API and it is isomorphic, which means that it can be used in both Node.js and browsers.
 *
 * - The body and headers of the request are automatically generated based on the body.
 * - The body of the response is dynamically parsed based on the Content-Type header, unless the parse option is set to false.
 * - Enabling logs will give you a detailed overview of the requests and responses.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/fetch)
 */
export declare class Fetch {
    /**
     * Sends a request.
     */
    static send<T, U, V>(input: FetchRequestInfo, init?: FetchRequestInit<V>): Promise<FetchResponse<T> | FetchError<U>>;
    /**
     * Sends a CONNECT request.
     */
    static connect<T, U>(input: FetchRequestInfo, init?: FetchRequestInit): Promise<FetchResponse<T> | FetchError<U>>;
    /**
     * Sends a DELETE request.
     */
    static delete<T, U, V>(input: FetchRequestInfo, body?: V, init?: FetchRequestInit): Promise<FetchResponse<T> | FetchError<U>>;
    /**
     * Sends a GET request.
     */
    static get<T, U>(input: FetchRequestInfo, init?: FetchRequestInit): Promise<FetchResponse<T> | FetchError<U>>;
    /**
     * Sends a HEAD request.
     */
    static head(input: FetchRequestInfo, init?: FetchRequestInit): Promise<FetchResponse | FetchError>;
    /**
     * Sends a OPTIONS request.
     */
    static options<T, U>(input: FetchRequestInfo, init?: FetchRequestInit): Promise<FetchResponse<T> | FetchError<U>>;
    /**
     * Sends a PATCH request.
     */
    static patch<T, U, V>(input: FetchRequestInfo, body?: V, init?: FetchRequestInit): Promise<FetchResponse<T> | FetchError<U>>;
    /**
     * Sends a POST request.
     */
    static post<T, U, V>(input: FetchRequestInfo, body?: V, init?: FetchRequestInit): Promise<FetchResponse<T> | FetchError<U>>;
    /**
     * Sends a PUT request.
     */
    static put<V>(input: FetchRequestInfo, body?: V, init?: FetchRequestInit): Promise<FetchResponse | FetchError>;
    /**
     * Sends a TRACE request.
     */
    static trace(input: FetchRequestInfo, init?: FetchRequestInit): Promise<FetchResponse | FetchError>;
}

declare interface FetchDecodeOptions {
    json?: DecodeJsonOptions;
    type?: FetchDecodeType;
}

export declare type FetchDecodeType = 'array-buffer' | 'blob' | 'form-data' | 'json' | 'text' | 'url-search-params';

declare interface FetchEncodeOptions {
    json?: EncodeJsonOptions;
}

/**
 * The FetchError class is used for errors that are returned by the Fetch class.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/fetch-error)
 */
export declare class FetchError<T = unknown> extends Error {
    /**
     * The response that caused the error.
     */
    response: FetchResponse<T>;
    constructor(error: Error, response: FetchResponse<T>);
    /**
     * Creates a new FetchError instance.
     */
    static from<T>(): FetchError<T>;
    static from<T>(error: Error): FetchError<T>;
    static from<T>(error: Error, response: FetchResponse<T>): FetchError<T>;
    static from<T>(response: FetchResponse<T>): FetchError<T>;
}

export declare type FetchRequestInfo = Request | string;

export declare interface FetchRequestInit<T = unknown> extends Omit<RequestInit, 'body'> {
    body?: T;
    decode?: FetchRequestInitDecode;
    encode?: FetchRequestInitEncode;
    logNativeOptions?: ToLoggableNativeFetchRequestInitOptions;
}

export declare type FetchRequestInitDecode = boolean | FetchDecodeOptions;

export declare type FetchRequestInitEncode = boolean | FetchEncodeOptions;

/**
 * The FetchResponse class is used for responses that are returned by the Fetch class.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/fetch-response)
 */
export declare class FetchResponse<T = unknown> implements Response {
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/body) */
    readonly body: ReadableStream<Uint8Array> | null;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/bodyUsed) */
    readonly bodyUsed: boolean;
    /**
     * The data that has been parsed.
     */
    data: T;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/headers) */
    readonly headers: Headers;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/ok) */
    readonly ok: boolean;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/redirected) */
    readonly redirected: boolean;
    protected readonly response: Response;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/status) */
    readonly status: number;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/statusText) */
    readonly statusText: string;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/type) */
    readonly type: ResponseType;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/url) */
    readonly url: string;
    constructor(response: Response, data?: T);
    /**
     * Decodes the body in the most appropriate way inferring the type from the content-type header.
     * Optionally the type can be specified, it can be array-buffer, blob, form-data, json, text or url-search-params.
     */
    decode(options?: FetchDecodeOptions): Promise<void>;
    decodeArrayBuffer(): Promise<void>;
    decodeBlob(): Promise<void>;
    decodeFormData(): Promise<void>;
    decodeJSON(options?: DecodeJsonOptions): Promise<void>;
    decodeText(): Promise<void>;
    decodeURLSearchParams(): Promise<void>;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/arrayBuffer) */
    arrayBuffer(): Promise<ArrayBuffer>;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/blob) */
    blob(): Promise<Blob>;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Response/clone) */
    clone(): Response;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/formData) */
    formData(): Promise<FormData>;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/json) */
    json(): Promise<any>;
    /** [MDN Reference](https://developer.mozilla.org/docs/Web/API/Request/text) */
    text(): Promise<string>;
    /**
     * Sets the data.
     */
    protected setData(data: any): void;
    /**
     * Creates a new FetchResponse instance.
     */
    static from<T>(data: T): FetchResponse<T>;
    static from<T>(response: Response): FetchResponse<T>;
    /**
     * Returns the content-type header.
     */
    protected get ContentType(): string;
}

/**
 * Flattens an object into a single-depth object with dot notation keys.
 * Optionally flattens arrays as well.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function flattenObject<T extends object>(object: T, options?: FlattenObjectOptions, parents?: string[]): Record<string, any>;

export declare interface FlattenObjectOptions {
    array: boolean;
}

/**
 * Returns a random string.
 *
 * - The alphabet is the set of characters that are used to generate the random string.
 * - The blacklist is a set of strings that are not allowed to be generated.
 * - The random function is a function that returns a random array of bytes.
 * - The prefix is the string that is prepended to the random string.
 * - The separator is the string that separates the prefix, the random string and the suffix.
 * - The size is the length of the random string.
 * - The suffix is the string that is appended to the random string.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function generateRandomString(options?: GenerateRandomStringOptions): string;

export declare interface GenerateRandomStringOptions {
    alphabet?: string;
    blacklist?: string[];
    prefix?: string;
    random?: (bytes: number) => Uint8Array;
    separator?: string;
    size?: number;
    suffix?: string;
}

export declare type GenerateRandomStringRandom = (bytes: number) => Uint8Array;

declare type Get = <T extends StorageItem>(key: string) => GetReturn<T>;

/**
 * Returns the absolute value of a number.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function getAbsoluteNumber(number: number): number;

/**
 * Returns the last item of an array.
 * Optionally you can pass a fallback value that will be returned if the array is empty.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/array)
 */
export declare function getArrayLastItem<T>(array: T[]): T | undefined;

export declare function getArrayLastItem<T>(array: T[], fallback: T): T;

/**
 * Returns the symmetric difference between two or more arrays.
 * Optionally you can pass a custom function to check if an item is included in the result array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/array)
 */
export declare function getArraysDifference<T>(arrays: T[][], predicate?: HasArrayItemPredicate<T>): T[];

/**
 * Returns the intersection between two or more arrays.
 * Optionally you can pass a custom function to check if an item is included in the result array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/array)
 */
export declare function getArraysIntersection<T>(arrays: T[][], predicate?: HasArrayItemPredicate<T>): T[];

/**
 * Returns a string in camel case.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function getCamelCaseString(string: string): string;

/**
 * Returns a string with the first letter capitalized.
 * Optionally the rest of the letters can be set to lowercase.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function getCapitalizedString(string: string, lowercase?: boolean): string;

/**
 * Returns the unix time of a date.
 * Optionally specify the unit of time, milliseconds or seconds.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/date)
 */
export declare function getDateUnixTime(date: string | number | Date, uot?: 'ms' | 's'): number;

/**
 * Returns the matching emoji flag from a country code.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/emoji)
 */
export declare function getEmojiFromCountryCode(code: string): string;

/**
 * Returns a header from a `FetchRequestInit` or `RequestInit` object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function getFetchRequestInitHeader<T>(init: FetchRequestInit<T> | RequestInit, name: string): string | null;

/**
 * Returns the headers entries from a `FetchRequestInit` or `RequestInit` object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function getFetchRequestInitHeadersEntries<T>(init: FetchRequestInit<T> | RequestInit): string[][];

/**
 * Returns the number with a fixed number of decimals.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function getFixedNumber(number: number, decimals: number): number;

/**
 * Returns the highest number in an array of numbers.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function getHighestNumber(numbers: number[]): number;

export declare function getHighestNumber(...numbers: number[]): number;

/**
 * Returns a string in kebab case.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function getKebabCaseString(string: string): string;

/**
 * Returns the number between a minimum and maximum value.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function getLimitedNumber(number: number, options?: GetLimitedNumberOptions): number;

export declare interface GetLimitedNumberOptions {
    max?: number;
    min?: number;
}

/**
 * Returns the ANSI color code for a logger level, the level can be `debug`, `error`, `info`, `verbose` or `warn`.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/logger)
 */
export declare function getLoggerAnsiColor(level: LoggerLevel): string;

/**
 * Returns the lowest number in an array of numbers.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function getLowestNumber(numbers: number[]): number;

export declare function getLowestNumber(...numbers: number[]): number;

/**
 * Returns the value of `process.env.NODE_ENV`.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function getNodeEnv(): string | undefined;

/**
 * Returns the percentage of a number between a minimum and maximum value.
 * Optionally the percentage can be rounded.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function getNumberPercentage(number: number, options?: GetNumberPercentageOptions): number;

export declare interface GetNumberPercentageOptions {
    max?: number;
    min?: number;
    round?: boolean;
}

/**
 * Returns the distance between two numbers.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function getNumbersDistance(a: number, b: number): number;

/**
 * Returns a property from an object. The key supports bracket and dot notation.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function getObjectProperty<T extends object, U>(object: T, key: KeyOf.Deep<T>): U | undefined;

export declare function getObjectProperty<T extends object, U>(object: T, key: KeyOf.Deep<T>, fallback: U): U;

export declare function getObjectProperty<T extends object, U>(object: T, key: string): U | undefined;

export declare function getObjectProperty<T extends object, U>(object: T, key: string, fallback: U): U;

/**
 * Returns a string in pascal case.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function getPascalCaseString(string: string): string;

/**
 * Returns the value of a key in `process.env`.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function getProcessEnvKey(key: string): string | undefined;

declare type GetReturn<T extends StorageItem> = T | Error | Promise<T | Error>;

/**
 * Returns a string in snake case.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function getSnakeCaseString(string: string): string;

/**
 * Returns a string with a symbol between each word.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function getSymbolCaseString(string: string, symbol: string): string;

/**
 * Returns a new TextDecoder or a stub if the environment does not support it.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/text)
 */
export declare function getTextDecoder(): TextDecoder;

/**
 * Returns a new TextEncoder or a stub if the environment does not support it.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/text)
 */
export declare function getTextEncoder(): TextEncoder;

/**
 * The `gql` function is a dirty hack to get editor features for GraphQL queries without using the real GraphQL parser.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/gql)
 */
export declare function gql(query: TemplateStringsArray): string;

/**
 * The GraphQlAPI class extends the RestAPI class and manages the requests to a GraphQL API.
 *
 * - The base URL of the API is automatically concatenated to the path of the requests.
 * - The config of the API is automatically merged with the config of the requests.
 * - The status of the requests is automatically tracked and can be accessed through the status property.
 * - The requests are sent with the Fetch class, so all features of the Fetch class are available.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/graphql-api)
 */
export declare class GraphQlAPI<T extends GraphQlApiConfig = GraphQlApiConfig, U = undefined> extends RestAPI<T, U> {
    /**
     * Sends a POST request to the GraphQL API.
     */
    post<V, W, X = U>(query: string, variables?: W, config?: T): Promise<FetchResponse<V> | FetchError<X>>;
    /**
     * Sends a mutation to the GraphQL API.
     */
    mutation<V, W extends object, X = U>(mutation: string, variables?: W, config?: T): Promise<GraphQlApiResponse<V> | FetchError<X>>;
    /**
     * Sends a query to the GraphQL API.
     */
    query<V, W extends object, X = U>(query: string, variables?: W, config?: T): Promise<GraphQlApiResponse<V> | FetchError<X>>;
}

export declare interface GraphQlApiConfig<T = unknown> extends RestApiConfig<T> {
}

export declare interface GraphQlApiRequestBody<T = object> {
    query: string;
    variables?: T | null;
}

export declare interface GraphQlApiResponse<T = any> extends FetchResponse<GraphQlApiResponseBody<T>> {
}

export declare interface GraphQlApiResponseBody<T = any> {
    data: T;
    errors?: any[];
}

export declare interface GraphQlApiResponseBodyError<T = any> {
    extensions?: T[];
    locations: GraphQlApiResponseBodyErrorLocation[];
    message: string;
}

export declare interface GraphQlApiResponseBodyErrorLocation {
    column: number;
    line: number;
}

declare type Has = (key: string) => HasReturn;

export declare type HasArrayItemPredicate<T> = (array: T[], item: T) => boolean;

/**
 * Checks if a `FetchRequestInit` or `RequestInit` object has a header.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function hasFetchRequestInitHeader<T>(init: FetchRequestInit<T> | RequestInit, name: string): boolean;

/**
 * Checks if an object has a property. The key supports bracket and dot notation.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function hasObjectProperty<T extends object>(object: T, key: KeyOf.Deep<T>): boolean;

export declare function hasObjectProperty<T extends object>(object: T, key: string): boolean;

/**
 * Checks if a key exists in `process.env`.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function hasProcessEnvKey(key: string): boolean;

declare type HasReturn = boolean | Error | Promise<boolean | Error>;

declare class History_2<T extends HistoryDataTarget = HistoryDataTarget, K extends keyof T = keyof T> {
    index: number;
    key: K;
    size: number;
    target: T;
    versions: T[K][];
    constructor(target: T, key: K, size?: number);
    redo(): void;
    undo(): void;
    push(): void;
    protected setIndex(offset: number): void;
    get isPushable(): boolean;
    get isNotPushable(): boolean;
    get isRedoable(): boolean;
    get isNotRedoable(): boolean;
    get isUndoable(): boolean;
    get isNotUndoable(): boolean;
}
export { History_2 as History }

export declare interface HistoryDataTarget extends Record<PropertyKey, any> {
}

/**
 * Imports `node-fetch` only if the Fetch API is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function importNodeFetch(): Promise<NodeFetch | Error>;

export declare type IntervalMapKey = bigint | number | string | symbol | Function;

export declare type IntervalMapValue = NodeJS.Timeout | number;

/**
 * Checks if the given value is an array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/array)
 */
export declare function isArray<T>(value: any): value is T[];

/**
 * Checks if `Blob` is defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isBlobDefined(): boolean;

/**
 * Checks if `Blob` is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isBlobNotDefined(): boolean;

/**
 * Checks if `document` is defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isDocumentDefined(): boolean;

/**
 * Checks if `document` is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isDocumentNotDefined(): boolean;

export declare type IsEqual<T1, T2> = (a: T1, b: T2) => boolean;

/**
 * Checks if an unknown value is an Error.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/error)
 */
export declare function isError(value: unknown): value is Error;

/**
 * Checks if `fetch` is defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isFetchDefined(): boolean;

/**
 * Checks if `fetch` is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isFetchNotDefined(): boolean;

/**
 * Checks if `File` is defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isFileDefined(): boolean;

/**
 * Checks if `File` is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isFileNotDefined(): boolean;

/**
 * Checks if `FormData` is defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isFormDataDefined(): boolean;

/**
 * Checks if `FormData` is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isFormDataNotDefined(): boolean;

/**
 * Checks if a value is an instance of a function.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/function)
 */
export declare function isInstanceOf<I extends Function>(value: any, instance: I): value is I;

/**
 * Checks if an interval is set.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/interval)
 */
export declare function isIntervalSet(key: IntervalMapKey): boolean;

/**
 * Checks if an interval is not set.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/interval)
 */
export declare function isIntervalUnset(key: IntervalMapKey): boolean;

/**
 * Checks if the `process.env.JEST_WORKER_ID` variable is defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isJestDefined(): boolean;

/**
 * Checks if the `process.env.JEST_WORKER_ID` variable is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isJestNotDefined(): boolean;

/**
 * Checks if the `process.env.NODE_ENV` variable is set to `development`.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isNodeEnvDevelopment(): boolean;

/**
 * Checks if the `process.env.NODE_ENV` variable is not set to `development`.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isNodeEnvNotDevelopment(): boolean;

/**
 * Checks if the `process.env.NODE_ENV` variable is not set to `production`.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isNodeEnvNotProduction(): boolean;

/**
 * Checks if the `process.env.NODE_ENV` variable is not set to `test`.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isNodeEnvNotTest(): boolean;

/**
 * Checks if the `process.env.NODE_ENV` variable is set to `production`.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isNodeEnvProduction(): boolean;

/**
 * Checks if the `process.env.NODE_ENV` variable is set to `test`.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isNodeEnvTest(): boolean;

/**
 * Checks if a value is not an Error.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/error)
 */
export declare function isNotError<T>(value: T | Error): value is T;

/**
 * Checks if a value is not an instance of a function.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/function)
 */
export declare function isNotInstanceOf<T, I extends Function>(value: T, instance: I): value is T;

/**
 * Checks if a unknown value is not a Promise.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/promise)
 */
export declare function isNotPromise<T>(value: T | Promise<T>): value is T;

/**
 * Checks if an unknown value is not a URL.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/url)
 */
export declare function isNotURL(value: unknown, base?: string | URL): boolean;

/**
 * Checks if the number is even.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function isNumberEven(number: number): boolean;

/**
 * Checks if the number is a multiple of another number.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function isNumberMultipleOf(number: number, of: number): boolean;

/**
 * Checks if the number is odd.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function isNumberOdd(number: number): boolean;

/**
 * Checks if an unknown value is an object. A value is considered an object if it is typeof "object", not null and not an array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isObject<T extends object>(value: unknown): value is T;

/**
 * Checks if an object is clonable. An object is considered clonable if it is an array or a plain object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isObjectClonable<T extends object>(object: T): boolean;

/**
 * Checks if an object is flattenable. An object is considered flattenable if it is an array or a plain object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isObjectFlattenable<T extends object>(object: T, options?: FlattenObjectOptions): boolean;

/**
 * Checks if an object has keys.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isObjectKeysPopulated<T extends object>(object: T): boolean;

/**
 * Checks if an object has values.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isObjectValuesPopulated<T extends object>(object: T): boolean;

/**
 * Checks if an unknown value is a plain object. A value is considered a plain object if it matches the default object prototype, it is typeof "object" and not null.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isPlainObject<T extends object>(value: unknown): value is T;

/**
 * Checks if `process` is defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isProcessDefined(): boolean;

/**
 * Checks if `process` is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isProcessNotDefined(): boolean;

/**
 * Checks if a unknown value is a Promise.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/promise)
 */
export declare function isPromise<T>(value: unknown): value is Promise<T>;

/**
 * Checks if a unknown value is a PromiseLike.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/promise)
 */
export declare function isPromiseLike<T>(value: unknown): value is PromiseLike<T>;

/**
 * Checks if a string is a float.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function isStringFloat(string: string): boolean;

/**
 * Checks if a string is an integer.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function isStringInt(string: string): boolean;

/**
 * Checks if a string is a JSON.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function isStringJSON(string: string): boolean;

/**
 * Checks if a string is not a JSON.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function isStringNotJSON(string: string): boolean;

/**
 * Checks if a string is not a URL.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function isStringNotURL(string: string): boolean;

/**
 * Checks if a string is a URL.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/string)
 */
export declare function isStringURL(string: string): boolean;

/**
 * Checks if `TextDecoder` is defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isTextDecoderDefined(): boolean;

/**
 * Checks if `TextDecoder` is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isTextDecoderNotDefined(): boolean;

/**
 * Checks if `TextEncoder` is defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isTextEncoderDefined(): boolean;

/**
 * Checks if `TextEncoder` is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isTextEncoderNotDefined(): boolean;

/**
 * Checks if a timeout is set.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/timeout)
 */
export declare function isTimeoutSet(key: TimeoutMapKey): boolean;

/**
 * Checks if a timeout is not set.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/timeout)
 */
export declare function isTimeoutUnset(key: TimeoutMapKey): boolean;

/**
 * Checks if an unknown value is a URL.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/url)
 */
export declare function isURL(value: unknown, base?: string | URL): boolean;

/**
 * Checks if `window` is defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isWindowDefined(): boolean;

/**
 * Checks if `window` is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/environment)
 */
export declare function isWindowNotDefined(): boolean;

/**
 * Joins a sequence of paths together.
 * Optionally uses the Windows standard.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/path)
 */
export declare function joinPaths(paths: string[], options?: JoinPathsOptions): string;

export declare function joinPaths(...paths: string[]): string;

declare interface JoinPathsOptions {
    windows?: boolean;
}

/**
 * Joins a sequence of paths together using the Windows standard.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/path)
 */
export declare function joinWindowsPaths(...paths: string[]): string;

export declare type JsonEncoding = 'utf-8';

export declare namespace KeyOf {
    export type Deep<T> = keyof T;
    export type DeepArray<T> = keyof T extends number ? keyof T : never;
    export type Shallow<T> = keyof T;
    export type ShallowArray<T> = keyof T extends number ? keyof T : never;
}

/**
 * The Localization class is used to localize anything that can be localized.
 *
 * - The language will persist in the storage, by default it will be stored in memory.
 * - The path of the localized string supports dot notation, for example: 'path.to.the.value'.
 * - The variables support dot notation as well and can be used inside the localized string, for example: 'Hello {name}!'.
 * - The instance also supports default variables, which can be overridden by the variables passed to the get method.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/localization)
 */
export declare class Localization<T extends LocalizationVariables = LocalizationVariables> {
    /**
     * The language that will be used to localize.
     */
    language: string;
    /**
     * The packs that will be used to localize.
     */
    packs: LocalizationPack[];
    /**
     * The storage that will be used to store the language.
     */
    storage: Storage_2;
    /**
     * The key that will be used to store the language.
     */
    storageKey: string;
    /**
     * The default variables that will be used to localize.
     */
    variables: T;
    constructor(init?: LocalizationInit<T>);
    /**
     * Retrieves the language from the storage and sets it.
     */
    initialize(): Promise<void | Error>;
    /**
     * Stores the language in the storage.
     */
    store(): Promise<void | Error>;
    /**
     * Adds the packs to the instance, if a pack for the language already exists, the data will be merged.
     */
    push(packs: LocalizationPack[]): void;
    push(...packs: LocalizationPack[]): void;
    /**
     * Retrieves the localized string from the pack.
     * Optionally you can pass variables that will be used to replace the variables inside the localized string.
     */
    get<T extends LocalizationVariables>(path: string, variables?: T): string;
    get<T extends LocalizationVariables>(language: string, path: string, variables?: T): string;
    /**
     * Sets the language.
     */
    setLanguage(language: string): void;
    /**
     * Sets the default variables.
     */
    setVariables(variables: T): void;
    /**
     * Checks if the pack has the path.
     */
    has(path: string): boolean;
    has(language: string, path: string): boolean;
    /**
     * Retrieves the pack by the language.
     */
    getPackByLanguage(language: string): LocalizationPack;
}

export declare interface LocalizationInit<T extends LocalizationVariables = LocalizationVariables> {
    language?: string;
    packs?: LocalizationPack[];
    storage?: {
        instance?: Storage_2;
        key?: string;
    };
    variables?: T;
}

export declare interface LocalizationPack {
    data: LocalizationPackData;
    language: string;
}

export declare interface LocalizationPackData {
    [key: string]: string | LocalizationPackData;
}

export declare interface LocalizationVariables extends Record<number | string, any> {
}

/**
 * The Logger class provides an isomorphic and consistent way to log messages to the console.
 *
 * - The level determines which messages are logged, the default level is `warn` on non-production environments and `error` on production environments.
 * - The status determines if the logger is enabled or disabled, the default status is `on` on non-test environments and `off` on test environments.
 * - The colors are on by default on non-browser environments and match the browser console colors.
 * - The separator is a string used to join the arguments passed to the logger methods, the default separator is a ` -> ` symbol.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/logger)
 */
export declare class Logger {
    /**
     * The colors are on by default on non-browser environments and match the browser console colors.
     */
    colors: boolean;
    /**
     * The level determines which messages are logged, it can be `verbose`, `debug`, `info`, `warn` or `error`.
     */
    level: LoggerLevel;
    /**
     * The name is used to get the level and status from the environment variables.
     */
    name: string;
    /**
     * The separator is a string used to join the arguments passed to the logger methods.
     */
    separator: string;
    /**
     * The status determines if the logger is enabled or disabled.
     */
    status: LoggerStatus;
    constructor(name: string, level?: LoggerLevel, status?: LoggerStatus, colors?: boolean, separator?: string);
    /**
     * Logs a message to the console if the logger is enabled and the level is above or equal to `verbose`.
     */
    verbose(...args: any[]): void;
    /**
     * Logs a message to the console if the logger is enabled and the level is above or equal to `debug`.
     */
    debug(...args: any[]): void;
    /**
     * Logs a message to the console if the logger is enabled and the level is above or equal to `info`.
     */
    info(...args: any[]): void;
    /**
     * Logs a message to the console if the logger is enabled and the level is above or equal to `warn`.
     */
    warn(...args: any[]): void;
    /**
     * Logs a message to the console if the logger is enabled and the level is above or equal to `error`.
     */
    error(...args: any[]): void;
    /**
     * Disables the logger.
     */
    disable(): void;
    /**
     * Enables the logger.
     */
    enable(): void;
    /**
     * Sets the level of the logger.
     */
    setLevel(level: LoggerLevel): void;
    /**
     * Sets the separator of the logger.
     */
    setSeparator(separator: string): void;
    /**
     * Disables the colors.
     */
    disableColors(): void;
    /**
     * Enables the colors.
     */
    enableColors(): void;
    /**
     * Formats the arguments passed to the logger methods, it joins the arguments with the separator and adds the colors if enabled.
     */
    format(level: LoggerLevel, ...args: any[]): any[];
    private formatArgs;
    private formatPrimitives;
    /**
     * Gets the level from the environment variables.
     */
    protected static getLevelFromEnvironment(name: string): LoggerLevel | undefined;
    /**
     * Gets the status from the environment variables.
     */
    protected static getStatusFromEnvironment(name: string): LoggerStatus | undefined;
    /**
     * Checks if the logger is disabled.
     */
    get isDisabled(): boolean;
    /**
     * Checks if the logger is enabled.
     */
    get isEnabled(): boolean;
    /**
     * Checks if the verbose level is disabled.
     */
    protected get isLevelVerboseDisabled(): boolean;
    /**
     * Checks if the debug level is disabled.
     */
    protected get isLevelDebugDisabled(): boolean;
    /**
     * Checks if the info level is disabled.
     */
    protected get isLevelInfoDisabled(): boolean;
    /**
     * Checks if the warn level is disabled.
     */
    protected get isLevelWarnDisabled(): boolean;
}

export declare const LOGGER_LEVELS: LoggerLevel[];

export declare const LOGGER_STATUSES: LoggerStatus[];

export declare type LoggerLevel = 'verbose' | 'debug' | 'info' | 'warn' | 'error';

export declare type LoggerStatus = 'off' | 'on';

/**
 * The `ma` function stands for `make async`. It takes a function and returns a function that returns a promise.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/ma)
 */
export declare function ma<T>(fn: (...args: any) => T | Promise<T>): (...args: any) => Promise<T>;

/**
 * The `MemoryStorage` is a `SyncStorage` that uses a `Map` as its storage, which means that it will only be available in the current session.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/storages/memory-storage)
 */
export declare const MemoryStorage: SyncStorage;

/**
 * Merges two or more `FetchRequestInit` or `RequestInit` objects.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function mergeFetchRequestInits<T>(target: FetchRequestInit<T>, ...sources: FetchRequestInit<T>[]): FetchRequestInit<T>;

/**
 * Merges two or more objects into a single object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function mergeObjects<T extends object, U extends object = T>(target: T, ...sources: Record<PropertyKey, any>[]): U;

/**
 * The `mtc` function stands for `make try catch`. It takes a function and returns a function that will return the result of the original function or an error.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/mtc)
 */
export declare function mtc<T>(fn: (...args: any) => T): (...args: any) => T | Error;

/**
 * The `mtcp` function stands for `make try catch promise`. It takes an async function and returns a function that will return the result of the original function or an error.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/mtcp)
 */
export declare function mtcp<T>(fn: (...args: any) => Promise<T>): (...args: any) => Promise<T | Error>;

export declare interface NodeFetch {
    default: any;
    Blob: typeof Blob;
    File: typeof File;
    FormData: typeof FormData;
    Headers: typeof Headers;
    Request: any;
    Response: any;
}

/**
 * The `noop` function stands for `no operation`. It takes any number of arguments and returns nothing.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/noop)
 */
export declare function noop(...args: any[]): any;

/**
 * Returns a new object without the properties that match the predicate or are in the list of keys. The keys support bracket and dot notation.
 * Optionally omits deep properties as well.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function omitObjectProperties<T extends object, U extends object = T>(object: T, predicate: OmitObjectPropertiesPredicate, options: OmitObjectPropertiesOptions & {
    deep: true;
}): U;

export declare function omitObjectProperties<T extends object, U extends object = T>(object: T, predicate: OmitObjectPropertiesPredicate<T>, options?: OmitObjectPropertiesOptions): U;

export declare function omitObjectProperties<T extends object, K extends KeyOf.Deep<T> = KeyOf.Deep<T>>(object: T, keys: K[]): Omit<T, K>;

export declare function omitObjectProperties<T extends object, K extends KeyOf.Shallow<T> = KeyOf.Shallow<T>>(object: T, keys: K[]): Omit<T, K>;

export declare function omitObjectProperties<T extends object, U extends object = T>(object: T, keys: string[]): U;

export declare interface OmitObjectPropertiesOptions {
    deep: boolean;
}

export declare type OmitObjectPropertiesPredicate<T extends object = object, K extends keyof T | string = keyof T | string> = (object: T, key: K, value: T extends Record<K, infer V> ? V : unknown, keys?: PropertyKey[]) => boolean;

/**
 * Parses an unknown value to a bigint.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function parseBigInt(value: unknown, fallback?: bigint): bigint;

/**
 * Parses an unknown value to a number.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/number)
 */
export declare function parseNumber(value: unknown, fallback?: number): number;

/**
 * Returns a new object with only the properties that match the predicate or are in the list of keys. The keys support bracket and dot notation.
 * Optionally picks deep properties as well.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function pickObjectProperties<T extends object, U extends object = T>(object: T, predicate: PickObjectPropertiesPredicate, options: PickObjectPropertiesOptions & {
    deep: true;
}): U;

export declare function pickObjectProperties<T extends object, U extends object = T>(object: T, predicate: PickObjectPropertiesPredicate, mode?: PickObjectPropertiesOptions): U;

export declare function pickObjectProperties<T extends object, K extends KeyOf.Deep<T> = KeyOf.Deep<T>>(object: T, keys: K[]): Pick<T, K>;

export declare function pickObjectProperties<T extends object, K extends KeyOf.Shallow<T> = KeyOf.Shallow<T>>(object: T, keys: K[]): Pick<T, K>;

export declare function pickObjectProperties<T extends object, U extends object = T>(object: T, keys: string[]): U;

export declare interface PickObjectPropertiesOptions {
    deep: boolean;
}

export declare type PickObjectPropertiesPredicate<T extends object = object, K extends keyof T | string = keyof T | string> = (object: T, key: K, value: T extends Record<K, infer V> ? V : unknown, keys?: PropertyKey[]) => boolean;

export declare type Primitive = bigint | boolean | null | number | string | symbol | undefined;

export declare type ProcessEnvValue = string | undefined;

export declare type PromiseState = 'fulfilled' | 'pending' | 'rejected';

export declare class Queue extends EventEmitter<QueueEvents> {
    protected concurrency: number;
    protected delay: number;
    protected processes: QueueProcess[];
    protected status: QueueStatus;
    protected timeout: number;
    constructor(options?: QueueOptions);
    start(): this;
    stop(): this;
    clear(): this;
    protected run(): void;
    protected runp(process: QueueProcess): void;
    push(fns: QueueFunction[]): this;
    push(...fns: QueueFunction[]): this;
    unshift(fns: QueueFunction[]): this;
    unshift(...fns: QueueFunction[]): this;
    getConcurrency(): number;
    getDelay(): number;
    getProcesses(): QueueProcess[];
    getStatus(): QueueStatus;
    getTimeout(): number;
    setConcurrency(concurrency: number): this;
    setDelay(delay: number): this;
    setTimeout(timeout: number): this;
    get isStatusRunning(): boolean;
    get isStatusStopped(): boolean;
}

export declare interface QueueEvents extends EventEmitterEvents {
    'process-fulfill': (process: QueueProcess<any>) => any;
    'process-reject': (process: QueueProcess) => any;
    'process-run': (process: QueueProcess) => any;
    'process-timeout': (process: QueueProcess) => any;
}

export declare type QueueFunction = () => Promise<unknown>;

export declare interface QueueOptions {
    autostart?: boolean;
    concurrency?: number;
    delay?: number;
    timeout?: number;
}

export declare interface QueueProcess<T = unknown> {
    fn: QueueFunction;
    id: string;
    reason?: any;
    status: QueueProcessStatus;
    value?: T;
}

export declare type QueueProcessStatus = 'pending' | 'running' | 'fulfilled' | 'rejected' | 'timed-out';

export declare type QueueStatus = 'running' | 'stopped';

/**
 * The `rc` functions stands for `return custom`. It takes a function and a value and returns the value.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/rc)
 */
export declare function rc<T>(fn: () => any, value: T): T;

/**
 * The `rcp` functions stands for `return custom promise`. It takes a function and a value and returns a promise that resolves to the value.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/rcp)
 */
export declare function rcp<T>(fn: () => any, value: T): Promise<T>;

declare type Remove = (key: string) => RemoveReturn;

/**
 * Removes all duplicates from an array.
 * Optionally you can pass a custom function to check if an item is included in the result array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/array)
 */
export declare function removeArrayDuplicates<T>(array: T[], predicate?: HasArrayItemPredicate<T>): T[];

/**
 * Removes items from an array that match the predicate or are in the items array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/array)
 */
export declare function removeArrayItems<T>(array: T[], predicate: RemoveArrayItemsPredicate<T>): T[];

export declare function removeArrayItems<T>(array: T[], items: T[]): T[];

export declare type RemoveArrayItemsPredicate<T> = (array: T[], item: T, items?: T[]) => boolean;

declare type RemoveReturn = void | Error | Promise<void | Error>;

/**
 * Removes the search params from an URL.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/url)
 */
export declare function removeSearchParamsFromURL(url: string): string;

export declare function removeSearchParamsFromURL(url: URL): URL;

export declare type RequestMethod = 'CONNECT' | 'DELETE' | 'GET' | 'HEAD' | 'OPTIONS' | 'PATCH' | 'POST' | 'PUT' | 'TRACE';

/**
 * The RestAPI class manages the requests to a REST API.
 *
 * - The base URL of the API is automatically concatenated to the path of the requests.
 * - The config of the API is automatically merged with the config of the requests.
 * - The status of the requests is automatically tracked and can be accessed through the status property.
 * - The requests are sent with the Fetch class, so all features of the Fetch class are available.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/rest-api)
 */
export declare class RestAPI<T extends RestApiConfig = RestApiConfig, U = undefined> {
    /**
     * The base URL of the REST API.
     */
    protected baseURL: string;
    /**
     * The default config of the REST API.
     */
    protected config: T;
    /**
     * The status of the requests.
     */
    readonly status: Status;
    constructor(baseURL?: string, config?: T);
    /**
     * Sends a request to the REST API.
     */
    protected send<V, W, X = U>(method: RequestMethod, path: string, body?: W, config?: T): Promise<FetchResponse<V> | FetchError<X>>;
    /**
     * Sends a CONNECT request to the REST API.
     */
    connect<V, W = U>(path: string, config?: T): Promise<FetchResponse<V> | FetchError<W>>;
    /**
     * Sends a DELETE request to the REST API.
     */
    delete<V, W = U>(path: string, config?: T): Promise<FetchResponse<V> | FetchError<W>>;
    /**
     * Sends a GET request to the REST API.
     */
    get<V, W = U>(path: string, config?: T): Promise<FetchResponse<V> | FetchError<W>>;
    /**
     * Sends a HEAD request to the REST API.
     */
    head(path: string, config?: T): Promise<FetchResponse | FetchError>;
    /**
     * Sends an OPTIONS request to the REST API.
     */
    options<V, W = U>(path: string, config?: T): Promise<FetchResponse<V> | FetchError<W>>;
    /**
     * Sends a PATCH request to the REST API.
     */
    patch<V, W, X = U>(path: string, body?: W, config?: T): Promise<FetchResponse<V> | FetchError<X>>;
    /**
     * Sends a POST request to the REST API.
     */
    post<V, W, X = U>(path: string, body?: W, config?: T): Promise<FetchResponse<V> | FetchError<X>>;
    /**
     * Sends a PUT request to the REST API.
     */
    put<V, W, X = U>(path: string, body?: W, config?: T): Promise<FetchResponse<V> | FetchError<X>>;
    /**
     * Sends a TRACE request to the REST API.
     */
    trace(path: string, config?: T): Promise<FetchResponse | FetchError>;
    /**
     * Sends a POST request if the mode is 'create' or a PUT request if the mode is 'update' to the REST API.
     */
    write<V, W, X = U>(mode: WriteMode, path: string, body?: W, config?: T): Promise<FetchResponse<V> | FetchError<X>>;
    /**
     * Transforms the body of the request.
     */
    transformBody<V>(method: RequestMethod, path: string, body: V | undefined, config: T): Promise<any>;
    /**
     * Transforms the query parameters of the request.
     */
    transformQueryParameters<V>(method: RequestMethod, path: string, body: V | undefined, config: T): Promise<string>;
    /**
     * Handles the error of the request.
     */
    handleError<V, W = U>(method: RequestMethod, path: string, body: V | undefined, config: T, error: FetchError<W>): Promise<boolean>;
    /**
     * Handles the pending state of the request.
     */
    handlePending<V>(method: RequestMethod, path: string, body: V | undefined, config: T): Promise<boolean>;
    /**
     * Handles the success of the request.
     */
    handleSuccess<V, W>(method: RequestMethod, path: string, body: W | undefined, config: T, response: FetchResponse<V>): Promise<boolean>;
    /**
     * Sets the status of the request.
     */
    protected setCallStatus(method: RequestMethod, path: string, config: T, status: string): void;
    getBaseURL(): string;
    getConfig(): T;
    setBaseURL(baseURL: string): this;
    setConfig(config: T): this;
    /**
     * Checks if the config status is settable.
     */
    protected isConfigStatusSettable(config: RestApiConfig, status: string): boolean;
}

export declare interface RestApiConfig<T = unknown> extends FetchRequestInit<T> {
    query?: object | string;
    status?: {
        blacklist?: string[];
        whitelist?: string[];
    };
}

/**
 * The `rne` function stands for `return new error`. It takes a message and options and returns a new error.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/rne)
 */
export declare function rne(message?: string, options?: any): Error;

/**
 * The `rv` function stands for `return void`. It takes a function and returns nothing.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/rv)
 */
export declare function rv(fn?: () => any): void;

/**
 * The `rvp` function stands for `return void promise`. It takes a function and returns a promise that resolves to nothing.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/rvp)
 */
export declare function rvp(fn?: () => any): Promise<void>;

/**
 * Serializes an `AracnaBlob` instance into a `Blob` object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/blob)
 */
export declare function serializeBlob(blob: AracnaBlob, endings?: EndingType): Blob;

/**
 * Serializes a cookie name-value pair into a string suitable for use in a `Set-Cookie` header or `document.cookie` property.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/cookie)
 */
export declare function serializeCookie(key: string, value: string, options?: CookieSerializeOptions): string | Error;

/**
 * Serializes an `AracnaFile` instance into a `File` object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/file)
 */
export declare function serializeFile(file: AracnaFile, endings?: EndingType): File;

/**
 * Serializes an object into a `FormData` object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/form-data)
 */
export declare function serializeFormData<T extends object>(object: T, options?: SerializeFormDataOptions): FormData;

declare interface SerializeFormDataOptions {
    json?: EncodeJsonOptions;
}

/**
 * Serializes an array, string or plain object to `URLSearchParams`.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/url)
 */
export declare function serializeURLSearchParams<T extends object>(params: SerializeURLSearchParamsInit<T>): URLSearchParams;

export declare type SerializeURLSearchParamsInit<T extends object = object> = string | string[][] | T | URLSearchParams;

export declare type SerializeURLSearchParamsType = 'string' | 'url-search-params';

declare type Set_2 = <T extends StorageItem>(key: string, item: T) => SetReturn;

/**
 * Sets a header in a `FetchRequestInit` or `RequestInit` object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function setFetchRequestInitHeader<T>(init: FetchRequestInit<T> | RequestInit, name: string, value: string): void;

/**
 * Sets a header a `FetchRequestInit` or `RequestInit` object if it is not set.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function setFetchRequestInitHeaderWhenUnset<T>(init: FetchRequestInit<T> | RequestInit, name: string, value: string): void;

/**
 * Sets an interval to run a function every `ms` milliseconds.
 *
 * Optionally the key can be specified, otherwise the function itself will be used as the key.
 * Optionally runs the function immediately with the `autorun` option.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/interval)
 */
declare function setInterval_2(fn: Function, ms: number, key?: IntervalMapKey, options?: SetIntervalOptions): void;
export { setInterval_2 as setInterval }

declare interface SetIntervalOptions {
    autorun?: boolean;
}

/**
 * Sets a property on an object. The key supports bracket and dot notation.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function setObjectProperty<T extends object, U>(object: T, key: KeyOf.Deep<T>, value: U): void | Error;

export declare function setObjectProperty<T extends object, U>(object: T, key: string, value: U): void | Error;

declare type SetReturn = void | Error | Promise<void | Error>;

/**
 * Sets a timeout to run a function after `ms` milliseconds.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/timeout)
 */
declare function setTimeout_2(fn: Function, ms: number, key?: TimeoutMapKey): void;
export { setTimeout_2 as setTimeout }

/**
 * The `sleep` function takes a number of milliseconds and returns a promise that resolves after that many milliseconds.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/sleep)
 */
export declare function sleep(ms: number): Promise<void>;

/**
 * The Status class manages the status of anything that can have a 4-state status: idle, pending, success, error.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/status)
 */
export declare class Status {
    /**
     * The Map that stores the status of each key.
     */
    readonly data: Map<string, string>;
    /**
     * The function that transforms the keys before storing them in the Map.
     */
    readonly transformer: StatusTransformer;
    constructor(transformer?: StatusTransformer);
    /**
     * Retrieves the status of the given keys.
     */
    get(...keys: string[]): string;
    /**
     * Sets the status of the given keys to idle.
     */
    idle(...keys: string[]): void;
    /**
     * Sets the status of the given keys to pending.
     */
    pending(...keys: string[]): void;
    /**
     * Sets the status of the given keys to success.
     */
    success(...keys: string[]): void;
    /**
     * Sets the status of the given keys to error.
     */
    error(...keys: string[]): void;
    /**
     * Sets the status of the given keys to the given status.
     */
    set(keys: string[], status: string): void;
    /**
     * Clears the status of every key.
     */
    clear(): void;
    /**
     * Checks if the status of the given keys is idle.
     */
    isIdle(...keys: string[]): boolean;
    /**
     * Checks if the status of the given keys is pending.
     */
    isPending(...keys: string[]): boolean;
    /**
     * Checks if the status of the given keys is success.
     */
    isSuccess(...keys: string[]): boolean;
    /**
     * Checks if the status of the given keys is error.
     */
    isError(...keys: string[]): boolean;
    /**
     * Checks if the status of every given key is idle.
     */
    isEveryIdle(...keys: string[][]): boolean;
    /**
     * Checks if the status of every given key is pending.
     */
    isEveryPending(...keys: string[][]): boolean;
    /**
     * Checks if the status of every given key is success.
     */
    isEverySuccess(...keys: string[][]): boolean;
    /**
     * Checks if the status of every given key is error.
     */
    isEveryError(...keys: string[][]): boolean;
    /**
     * Checks if the status of some given key is idle.
     */
    areSomeIdle(...keys: string[][]): boolean;
    /**
     * Checks if the status of some given key is pending.
     */
    areSomePending(...keys: string[][]): boolean;
    /**
     * Checks if the status of some given key is success.
     */
    areSomeSuccess(...keys: string[][]): boolean;
    /**
     * Checks if the status of some given key is error.
     */
    areSomeError(...keys: string[][]): boolean;
    static get IDLE(): string;
    static get PENDING(): string;
    static get SUCCESS(): string;
    static get ERROR(): string;
}

export declare type StatusTransformer = (keys: string[]) => string;

declare type Storage_2 = AsyncStorage | SyncStorage;
export { Storage_2 as Storage }

declare class Storage_3 {
    /**
     * The storage instance name.
     */
    readonly name: string;
    protected readonly _clear: Clear;
    protected readonly _get: Get;
    protected readonly _has: Has;
    protected readonly _remove: Remove;
    protected readonly _set: Set_2;
    constructor(name: string, clear: Clear, get: Get, has: Has, remove: Remove, set: Set_2);
    protected clear_(cleared: void | Error): void | Error;
    clear(): ClearReturn;
    protected get_<T extends StorageItem>(key: string, item: T | Error): T | Error;
    get<T extends StorageItem>(key: string): GetReturn<T>;
    protected remove_(key: string, removed: void | Error): void | Error;
    protected remove__<T extends StorageItem>(key: string, keys: KeyOf.Deep<T>[], item: T | Error): T | undefined;
    protected remove___<T extends StorageItem>(key: string, item: T, set: void | Error): void | Error;
    remove<T extends StorageItem>(key: string, keys?: KeyOf.Deep<T>[]): RemoveReturn;
    protected set_<T extends StorageItem>(key: string, item: T | Error, set: void | Error): void | Error;
    protected set__<T extends StorageItem>(item: T, keys: KeyOf.Deep<T>[], current: T | Error): T | Error;
    set<T extends StorageItem>(key: string, item: T, keys?: KeyOf.Deep<T>[]): SetReturn;
    protected copy_<T1 extends StorageItem, T2 extends StorageTarget, T extends T1 & T2>(key: string, target: T2, keys: KeyOf.Deep<T>[] | undefined, item: T1 | Error): void | Error;
    copy<T1 extends StorageItem, T2 extends StorageTarget, T extends T1 & T2>(key: string, target: T2, keys?: KeyOf.Deep<T>[]): CopyReturn;
    protected has_<T extends StorageItem>(keys: KeyOf.Deep<T>[] | undefined, item: T | Error): boolean;
    has<T extends StorageItem>(key: string, keys?: KeyOf.Deep<T>[]): HasReturn;
}

export declare interface StorageItem extends Record<PropertyKey, any> {
}

export declare interface StorageTarget extends Record<PropertyKey, any> {
}

export declare const STUB_COOKIE_GET: (map: Map<string, string>) => () => string;

export declare const STUB_COOKIE_SET: (map: Map<string, string>, deserialize?: Function) => (cookie: string) => void;

export declare const STUB_STORAGE: (map: Map<string, string>) => Storage;

export declare const STUB_TEXT_DECODER: TextDecoder;

export declare const STUB_TEXT_ENCODER: TextEncoder;

export declare class StubBlob {
    readonly size: number;
    readonly type: string;
    constructor(blobParts?: BlobPart[], options?: BlobPropertyBag);
    arrayBuffer(): Promise<ArrayBuffer>;
    slice(start?: number, end?: number, contentType?: string): this;
    stream(): ReadableStream<Uint8Array>;
    text(): Promise<string>;
}

export declare class StubFile extends StubBlob {
    readonly lastModified: number;
    readonly name: string;
    readonly webkitRelativePath: string;
    constructor(fileBits: BlobPart[], fileName: string, options?: FilePropertyBag);
}

/**
 * The SyncStorage class is an abstraction to implement any synchronous storage API in an uniform way.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/sync-storage)
 */
export declare class SyncStorage extends Storage_3 {
    constructor(name: string, clear: () => void, get: <T extends StorageItem>(key: string) => T, has: (key: string) => boolean, remove: (key: string) => void, set: <T extends StorageItem>(key: string, item: T) => void);
    /**
     * Clears the storage, removing all the items.
     */
    clear(): void | Error;
    /**
     * Retrieves an item from the storage.
     */
    get<T extends StorageItem>(key: string): T | Error;
    /**
     * Removes an item from the storage.
     * Optionally you can specify the keys of the item that you want to remove, if you don't specify any key the whole item will be removed.
     */
    remove<T extends StorageItem>(key: string, keys?: (keyof T)[] | undefined): void | Error;
    /**
     * Sets an item in the storage.
     * Optionally you can specify the keys of the item that you want to set, if you don't specify any key the whole item will be set.
     */
    set<T extends StorageItem>(key: string, item: T, keys?: (keyof T)[] | undefined): void | Error;
    /**
     * Copies an item from the storage to a target object.
     * Optionally you can specify the keys of the item that you want to copy, if you don't specify any key the whole item will be copied.
     */
    copy<T1 extends StorageItem, T2 extends StorageTarget = StorageTarget, T extends T1 & T2 = T1 & T2>(key: string, target: T2, keys?: (keyof T)[] | undefined): void | Error;
    /**
     * Checks if an item exists in the storage.
     * Optionally you can specify the keys of the item that you want to check, if you don't specify any key the whole item will be checked.
     */
    has<T extends StorageItem>(key: string, keys?: (keyof T)[] | undefined): boolean;
}

/**
 * The `tc` function stands for `try catch`. It takes a function and returns the result of the function or the error that was thrown.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/tc)
 */
export declare function tc<T, U extends Error = Error>(fn: () => T, log?: boolean): T | U;

/**
 * The `tcp` function stands for `try catch promise`. It takes a function and returns a promise that resolves to the result of the function or the error that was thrown.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/tcp)
 */
export declare function tcp<T, U extends Error = Error>(fn: () => Promise<T>, log?: boolean): Promise<T | U>;

export declare type Theme = 'dark' | 'light' | 'system';

/**
 * The `throttle` function is used to prevent a function from being called too many times in a short period.
 * The function will only be called if the time since the last call is greater than or equal to the specified amount of time.
 *
 * Optionally the key can be specified, otherwise the function itself will be used as the key.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/throttle)
 */
export declare function throttle(fn: Function, ms: number, key?: ThrottleMapKey): void;

export declare type ThrottleMapKey = bigint | number | string | symbol | Function;

/**
 * The `tie` function stands for `throw if error`. It takes a value and throws an error if the value is an error.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/tie)
 */
export declare function tie<T, U extends Error = Error>(value: T | U): T;

export declare type TimeoutMapKey = bigint | number | string | symbol | Function;

export declare type TimeoutMapValue = NodeJS.Timeout | number;

/**
 * The `tne` function stands for `throw new error`. It takes a message and options and throws a new error.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/tne)
 */
export declare function tne(message?: string, options?: any): void;

/**
 * Returns a version of a `FetchRequestInit` object that is easier to read in logs.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function toLoggableFetchRequestInit<T>(init: FetchRequestInit<T>, options?: ToLoggableFetchRequestInitOptions): FetchRequestInit;

declare interface ToLoggableFetchRequestInitOptions {
    deserializeFormData?: DeserializeFormDataOptions;
    deserializeURLSearchParamsType?: DeserializeURLSearchParamsType;
}

/**
 * Returns a version of a `RequestInit` object that is easier to read in logs.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function toLoggableNativeFetchRequestInit(init: RequestInit, options?: ToLoggableNativeFetchRequestInitOptions): RequestInit;

declare interface ToLoggableNativeFetchRequestInitOptions extends ToLoggableFetchRequestInitOptions {
    decodeJSON?: DecodeJsonOptions;
}

/**
 * Converts a `FetchRequestInit` object to a `RequestInit` object.
 * Sets the `content-type` header based on the type of the body.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function toNativeFetchRequestInit<T>(init: FetchRequestInit<T>): RequestInit;

/**
 * The Typeahead class is a state and options holder for the typeahead function.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/typeahead)
 */
export declare class Typeahead<T> extends EventEmitter<TypeaheadEvents<T>> {
    /**
     * The chunks that make up the query string.
     */
    protected chunks: string[];
    /**
     * The debounce time.
     */
    protected debounceTime: number;
    /**
     * The items to search.
     */
    protected items: T[];
    /**
     * The key of the instance.
     */
    protected readonly key: TypeaheadMapKey;
    /**
     * The predicate function.
     */
    protected predicate: TypeaheadPredicate<T>;
    constructor(key: TypeaheadMapKey, items?: T[], predicate?: TypeaheadPredicate<T>, debounceTime?: number);
    /**
     * The debounce function.
     */
    protected debouncefn: () => void;
    /**
     * Searches the items for a match and emits the match event if a match is found.
     */
    search(): this;
    /**
     * Returns the debounce time.
     */
    getDebounceTime(): number;
    /**
     * Returns the items.
     */
    getItems(): T[];
    /**
     * Returns the key of the instance.
     */
    getKey(): TypeaheadMapKey;
    /**
     * Returns the predicate function.
     */
    getPredicate(): TypeaheadPredicate<T>;
    /**
     * Returns the query string.
     */
    getQuery(): string;
    /**
     * Pushes chunks to the chunks array.
     */
    pushChunks(...chunks: string[]): this;
    /**
     * Sets the debounce time.
     */
    setDebounceTime(debounceTime: number | undefined): this;
    /**
     * Sets the items.
     */
    setItems(items: T[] | undefined): this;
    /**
     * Sets the chunks.
     */
    setChunks(chunks: string[] | undefined): this;
    /**
     * Sets the predicate function.
     */
    setPredicate(predicate: TypeaheadPredicate<T> | undefined): this;
}

/**
 * The `typeahead` function is used to search for items in a list, based on user input.
 * When a match is found, the `match` event is emitted.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/typeahead)
 */
export declare function typeahead<T>(key: TypeaheadMapKey, chunks: string | string[], options?: TypeaheadOptions<T>): Typeahead<T>;

export declare interface TypeaheadEvents<T> extends EventEmitterEvents {
    match: (item: T) => any;
}

export declare type TypeaheadMapKey = bigint | number | string | symbol;

export declare type TypeaheadMapValue = Typeahead<any>;

export declare type TypeaheadOnMatch<T> = (item: T) => any;

export declare interface TypeaheadOptions<T> {
    debounceTime?: number;
    items?: T[];
    listeners?: EventEmitterListener<TypeaheadEvents<T>>[];
    predicate?: TypeaheadPredicate<T>;
}

export declare type TypeaheadPredicate<T> = (item: T, query: string, index: number, items: T[]) => unknown;

export declare type URLSearchParamsRecord = Record<string, string>;

/**
 * Polyfills the Fetch API with `node-fetch` if the Fetch API is not defined.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/fetch)
 */
export declare function useNodeFetch(NodeFetch: NodeFetch | Error): Promise<void>;

/**
 * The VisibilityController class is used to control the visibility of anything that can be hidden or shown.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/visibility-controller)
 */
export declare class VisibilityController {
    protected readonly data: Map<string, string>;
    /**
     * Hides the entity with the given name.
     * Optionally a delay can be set.
     */
    hide(name: string, delay?: number): Promise<void>;
    /**
     * Shows the entity with the given name.
     * Optionally a delay can be set.
     */
    show(name: string, delay?: number): Promise<void>;
    /**
     * Toggles the entity with the given name.
     * Optionally a delay can be set, which can be different for hiding and showing.
     */
    toggle(name: string, delay?: VisibilityControllerToggleDelay): Promise<void>;
    /**
     * Clears the visibility of all entities.
     */
    clear(): void;
    protected get(name: string): string;
    /**
     * Checks if the entity with the given name is hidden.
     */
    isHidden(name: string): boolean;
    /**
     * Checks if the entity with the given name is hiding.
     */
    isHiding(name: string): boolean;
    /**
     * Checks if the entity with the given name is showing.
     */
    isShowing(name: string): boolean;
    /**
     * Checks if the entity with the given name is visible.
     */
    isVisible(name: string): boolean;
    /**
     * Checks if any entity is hidden.
     */
    get hasHidden(): boolean;
    /**
     * Checks if any entity is hiding.
     */
    get hasHiding(): boolean;
    /**
     * Checks if any entity is showing.
     */
    get hasShowing(): boolean;
    /**
     * Checks if any entity is visible.
     */
    get hasVisible(): boolean;
    protected static get HIDDEN(): string;
    protected static get HIDING(): string;
    protected static get SHOWING(): string;
    protected static get VISIBLE(): string;
}

export declare type VisibilityControllerToggleDelay = number | {
    hide?: number;
    show?: number;
};

/**
 * The `wf` function stands for `wait for`. It takes a function and waits for it to return a truthy value or an error.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/wf)
 */
export declare function wf(fn: () => any, ms?: number, timeout?: number): Promise<void | Error>;

/**
 * The `wfp` function stands for `wait for promise`. It takes a function that returns a promise and waits for it to resolve to a truthy value or an error.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/functions/wfp)
 */
export declare function wfp(fn: () => Promise<any>, ms?: number, timeout?: number): Promise<void | Error>;

export declare interface WithWriteMode {
    mode?: WriteMode;
    isModeCreate?: boolean;
    isModeUpdate?: boolean;
}

export declare type WriteMode = 'create' | 'update';

export { }
