import type { PromiseState } from '../definitions/types.js';
/**
 * The DeferredPromise class is built on top of the native Promise class. It provides a way to resolve or reject a promise from the outside.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/deferred-promise)
 */
export declare class DeferredPromise<T> {
    /**
     * The native promise instance.
     */
    readonly instance: Promise<T>;
    /**
     * The reason for the rejection of the promise.
     */
    reason?: any;
    /**
     * The state of the promise, can be pending, fulfilled or rejected.
     */
    state: PromiseState;
    /**
     * The value of the resolved promise.
     */
    value?: T;
    private _reject;
    private _resolve;
    constructor();
    reject: (reason?: any) => void;
    /**
     * Resolves the promise with a value or the result of another promise.
     */
    resolve: (value: T | PromiseLike<T>) => void;
    /**
     * Attaches a callback for only the rejection of the Promise.
     * @param onrejected The callback to execute when the Promise is rejected.
     * @returns A Promise for the completion of the callback.
     */
    catch<TResult = never>(onrejected?: ((reason: any) => TResult | PromiseLike<TResult>) | undefined | null): this;
    /**
     * Attaches a callback that is invoked when the Promise is settled (fulfilled or rejected). The
     * resolved value cannot be modified from the callback.
     * @param onfinally The callback to execute when the Promise is settled (fulfilled or rejected).
     * @returns A Promise for the completion of the callback.
     */
    finally(onfinally?: (() => void) | undefined | null): this;
    /**
     * Attaches callbacks for the resolution and/or rejection of the Promise.
     * @param onfulfilled The callback to execute when the Promise is resolved.
     * @param onrejected The callback to execute when the Promise is rejected.
     * @returns A Promise for the completion of which ever callback is executed.
     */
    then<TResult1 = T, TResult2 = never>(onfulfilled?: ((value: T) => TResult1 | PromiseLike<TResult1>) | undefined | null, onrejected?: ((reason: any) => TResult2 | PromiseLike<TResult2>) | undefined | null): this;
    /**
     * Checks if the promise is fulfilled.
     */
    get isFulfilled(): boolean;
    /**
     * Checks if the promise is pending.
     */
    get isPending(): boolean;
    /**
     * Checks if the promise is rejected.
     */
    get isRejected(): boolean;
    /**
     * Checks if the promise is resolved.
     */
    get isResolved(): boolean;
}
