import { CloneObjectOptions, DeleteObjectPropertiesOptions, FlattenObjectOptions, OmitObjectPropertiesOptions, PickObjectPropertiesOptions } from '../definitions/interfaces.js';
import { DeleteObjectPropertiesPredicate, KeyOf, OmitObjectPropertiesPredicate, PickObjectPropertiesPredicate } from '../definitions/types.js';
/**
 * Creates a copy of an object.
 * Optionally the copy can be deep.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function cloneObject<T extends object>(object: T, options?: CloneObjectOptions): T;
/**
 * Copies a property from one object to another. The key supports bracket and dot notation.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function copyObjectProperty<T1 extends object, T2 extends object, T extends T1 & T2>(source: T, key: KeyOf.Deep<T>, target: T): void | Error;
export declare function copyObjectProperty<T1 extends object, T2 extends object, T extends T1 & T2>(source: T, key: string, target: T): void | Error;
/**
 * Deletes a property from an object. The key supports bracket and dot notation.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function deleteObjectProperty<T extends object>(object: T, key: KeyOf.Deep<T>): void;
export declare function deleteObjectProperty<T extends object>(object: T, key: string): void;
/**
 * Deletes the properties of an object that either match the predicate or are in the list of keys. The keys support bracket and dot notation.
 * Optionally deletes deep properties as well.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function deleteObjectProperties<T extends object>(object: T, predicate: DeleteObjectPropertiesPredicate, options: DeleteObjectPropertiesOptions & {
    deep: true;
}): void;
export declare function deleteObjectProperties<T extends object>(object: T, predicate: DeleteObjectPropertiesPredicate<T>, options?: DeleteObjectPropertiesOptions): void;
export declare function deleteObjectProperties<T extends object>(object: T, keys: KeyOf.Deep<T>[]): void;
export declare function deleteObjectProperties<T extends object>(object: T, keys: KeyOf.Shallow<T>[]): void;
export declare function deleteObjectProperties<T extends object>(object: T, keys: string[]): void;
/**
 * Flattens an object into a single-depth object with dot notation keys.
 * Optionally flattens arrays as well.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function flattenObject<T extends object>(object: T, options?: FlattenObjectOptions, parents?: string[]): Record<string, any>;
/**
 * Returns a property from an object. The key supports bracket and dot notation.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function getObjectProperty<T extends object, U>(object: T, key: KeyOf.Deep<T>): U | undefined;
export declare function getObjectProperty<T extends object, U>(object: T, key: KeyOf.Deep<T>, fallback: U): U;
export declare function getObjectProperty<T extends object, U>(object: T, key: string): U | undefined;
export declare function getObjectProperty<T extends object, U>(object: T, key: string, fallback: U): U;
/**
 * Merges two or more objects into a single object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function mergeObjects<T extends object, U extends object = T>(target: T, ...sources: Record<PropertyKey, any>[]): U;
/**
 * Returns a new object without the properties that match the predicate or are in the list of keys. The keys support bracket and dot notation.
 * Optionally omits deep properties as well.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function omitObjectProperties<T extends object, U extends object = T>(object: T, predicate: OmitObjectPropertiesPredicate, options: OmitObjectPropertiesOptions & {
    deep: true;
}): U;
export declare function omitObjectProperties<T extends object, U extends object = T>(object: T, predicate: OmitObjectPropertiesPredicate<T>, options?: OmitObjectPropertiesOptions): U;
export declare function omitObjectProperties<T extends object, K extends KeyOf.Deep<T> = KeyOf.Deep<T>>(object: T, keys: K[]): Omit<T, K>;
export declare function omitObjectProperties<T extends object, K extends KeyOf.Shallow<T> = KeyOf.Shallow<T>>(object: T, keys: K[]): Omit<T, K>;
export declare function omitObjectProperties<T extends object, U extends object = T>(object: T, keys: string[]): U;
/**
 * Returns a new object with only the properties that match the predicate or are in the list of keys. The keys support bracket and dot notation.
 * Optionally picks deep properties as well.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function pickObjectProperties<T extends object, U extends object = T>(object: T, predicate: PickObjectPropertiesPredicate, options: PickObjectPropertiesOptions & {
    deep: true;
}): U;
export declare function pickObjectProperties<T extends object, U extends object = T>(object: T, predicate: PickObjectPropertiesPredicate, mode?: PickObjectPropertiesOptions): U;
export declare function pickObjectProperties<T extends object, K extends KeyOf.Deep<T> = KeyOf.Deep<T>>(object: T, keys: K[]): Pick<T, K>;
export declare function pickObjectProperties<T extends object, K extends KeyOf.Shallow<T> = KeyOf.Shallow<T>>(object: T, keys: K[]): Pick<T, K>;
export declare function pickObjectProperties<T extends object, U extends object = T>(object: T, keys: string[]): U;
/**
 * Sets a property on an object. The key supports bracket and dot notation.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function setObjectProperty<T extends object, U>(object: T, key: KeyOf.Deep<T>, value: U): void | Error;
export declare function setObjectProperty<T extends object, U>(object: T, key: string, value: U): void | Error;
/**
 * Checks if an object has a property. The key supports bracket and dot notation.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function hasObjectProperty<T extends object>(object: T, key: KeyOf.Deep<T>): boolean;
export declare function hasObjectProperty<T extends object>(object: T, key: string): boolean;
/**
 * Checks if an unknown value is an object. A value is considered an object if it is typeof "object", not null and not an array.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isObject<T extends object>(value: unknown): value is T;
/**
 * Checks if an object is clonable. An object is considered clonable if it is an array or a plain object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isObjectClonable<T extends object>(object: T): boolean;
/**
 * Checks if an object is flattenable. An object is considered flattenable if it is an array or a plain object.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isObjectFlattenable<T extends object>(object: T, options?: FlattenObjectOptions): boolean;
/**
 * Checks if an object has keys.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isObjectKeysPopulated<T extends object>(object: T): boolean;
/**
 * Checks if an object has values.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isObjectValuesPopulated<T extends object>(object: T): boolean;
/**
 * Checks if an unknown value is a plain object. A value is considered a plain object if it matches the default object prototype, it is typeof "object" and not null.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/utils/object)
 */
export declare function isPlainObject<T extends object>(value: unknown): value is T;
