import { CookieParseOptions, CookieSerializeOptions } from 'cookie';
import { CookieItem, CookieObject } from '../definitions/interfaces.js';
import { KeyOf, Primitive } from '../definitions/types.js';
/**
 * The Cookie class is an abstraction to implement any cookie API in an uniform way.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/cookie)
 */
export declare class Cookie {
    /**
     * The cookie instance name.
     */
    readonly name: string;
    /**
     * The cookie separator used to separate the cookie name from the cookie item key.
     */
    readonly separator: string;
    private readonly _get;
    private readonly _set;
    constructor(name: string, get: () => string, set: (cookies: string) => void, separator?: string);
    /**
     * Clears all cookies.
     */
    clear(options?: CookieSerializeOptions): void | Error;
    /**
     * Retrieves an item from the cookies.
     */
    get<T extends CookieItem>(key: string): T | Error;
    /**
     * Removes an item from the cookies.
     * Optionally you can specify the keys of the item that you want to remove, if you don't specify any key the whole item will be removed.
     */
    remove<T extends CookieItem>(key: string, options?: CookieSerializeOptions, keys?: KeyOf.Shallow<T>[]): void | Error;
    /**
     * Sets an item in the cookies.
     * Optionally you can specify the keys of the item that you want to set, if you don't specify any key the whole item will be set.
     */
    set<T extends CookieItem>(key: string, item: T, options?: CookieSerializeOptions, keys?: KeyOf.Shallow<T>[]): void | Error;
    /**
     * Copies an item from the cookies to a target object.
     * Optionally you can specify the keys of the item that you want to copy, if you don't specify any key the whole item will be copied.
     */
    copy<T1 extends CookieItem, T2 extends CookieItem = CookieItem, T extends T1 & T2 = T1 & T2>(key: string, target: T2, keys?: KeyOf.Shallow<T>[]): void | Error;
    /**
     * Checks if an item exists in the cookies.
     * Optionally you can specify the keys of the item that you want to check, if you don't specify any key the whole item will be checked.
     */
    has<T extends CookieItem>(key: string, keys?: KeyOf.Shallow<T>[]): boolean;
    protected deserialize(options?: CookieParseOptions): CookieObject | Error;
    protected serialize(key: string, value: string, options?: CookieSerializeOptions): string | Error;
    protected serialize<T extends CookieItem>(key: string, ik: keyof T, value: Primitive, options?: CookieSerializeOptions): string | Error;
    protected toCookieItemKey<T extends CookieItem>(key: string, ik: KeyOf.Shallow<T>): string;
    protected toDocumentCookieName<T extends CookieItem>(key: string, ik: KeyOf.Shallow<T>): string;
}
