import { StorageItem, StorageTarget } from '../definitions/interfaces.js';
import { KeyOf } from '../definitions/types.js';
import { Storage } from './storage.js';
/**
 * The AsyncStorage class is an abstraction to implement any asynchronous storage API in an uniform way.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/async-storage)
 */
export declare class AsyncStorage extends Storage {
    constructor(name: string, clear: () => Promise<void>, get: <T extends StorageItem>(key: string) => Promise<T>, has: (key: string) => Promise<boolean>, remove: (key: string) => Promise<void>, set: <T extends StorageItem>(key: string, item: T) => Promise<void>);
    /**
     * Clears the storage, removing all the items.
     */
    clear(): Promise<void | Error>;
    /**
     * Retrieves an item from the storage.
     */
    get<T extends StorageItem>(key: string): Promise<T | Error>;
    /**
     * Removes an item from the storage.
     * Optionally you can specify the keys of the item that you want to remove, if you don't specify any key the whole item will be removed.
     */
    remove<T extends StorageItem>(key: string, keys?: KeyOf.Deep<T>[]): Promise<void | Error>;
    /**
     * Sets an item in the storage.
     * Optionally you can specify the keys of the item that you want to set, if you don't specify any key the whole item will be set.
     */
    set<T extends StorageItem>(key: string, item: T, keys?: KeyOf.Deep<T>[]): Promise<void | Error>;
    /**
     * Copies an item from the storage to a target object.
     * Optionally you can specify the keys of the item that you want to copy, if you don't specify any key the whole item will be copied.
     */
    copy<T1 extends StorageItem, T2 extends StorageTarget = StorageTarget, T extends T1 & T2 = T1 & T2>(key: string, target: T2, keys?: KeyOf.Deep<T>[]): Promise<void | Error>;
    /**
     * Checks if an item exists in the storage.
     * Optionally you can specify the keys of the item that you want to check, if you don't specify any key the whole item will be checked.
     */
    has<T extends StorageItem>(key: string, keys?: KeyOf.Deep<T>[]): Promise<boolean>;
}
