import { LoggerLevel, LoggerStatus } from '../definitions/types.js';
/**
 * The Logger class provides an isomorphic and consistent way to log messages to the console.
 *
 * - The level determines which messages are logged, the default level is `warn` on non-production environments and `error` on production environments.
 * - The status determines if the logger is enabled or disabled, the default status is `on` on non-test environments and `off` on test environments.
 * - The colors are on by default on non-browser environments and match the browser console colors.
 * - The separator is a string used to join the arguments passed to the logger methods, the default separator is a ` -> ` symbol.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/logger)
 */
export declare class Logger {
    /**
     * The colors are on by default on non-browser environments and match the browser console colors.
     */
    colors: boolean;
    /**
     * The level determines which messages are logged, it can be `verbose`, `debug`, `info`, `warn` or `error`.
     */
    level: LoggerLevel;
    /**
     * The name is used to get the level and status from the environment variables.
     */
    name: string;
    /**
     * The separator is a string used to join the arguments passed to the logger methods.
     */
    separator: string;
    /**
     * The status determines if the logger is enabled or disabled.
     */
    status: LoggerStatus;
    constructor(name: string, level?: LoggerLevel, status?: LoggerStatus, colors?: boolean, separator?: string);
    /**
     * Logs a message to the console if the logger is enabled and the level is above or equal to `verbose`.
     */
    verbose(...args: any[]): void;
    /**
     * Logs a message to the console if the logger is enabled and the level is above or equal to `debug`.
     */
    debug(...args: any[]): void;
    /**
     * Logs a message to the console if the logger is enabled and the level is above or equal to `info`.
     */
    info(...args: any[]): void;
    /**
     * Logs a message to the console if the logger is enabled and the level is above or equal to `warn`.
     */
    warn(...args: any[]): void;
    /**
     * Logs a message to the console if the logger is enabled and the level is above or equal to `error`.
     */
    error(...args: any[]): void;
    /**
     * Disables the logger.
     */
    disable(): void;
    /**
     * Enables the logger.
     */
    enable(): void;
    /**
     * Sets the level of the logger.
     */
    setLevel(level: LoggerLevel): void;
    /**
     * Sets the separator of the logger.
     */
    setSeparator(separator: string): void;
    /**
     * Disables the colors.
     */
    disableColors(): void;
    /**
     * Enables the colors.
     */
    enableColors(): void;
    /**
     * Formats the arguments passed to the logger methods, it joins the arguments with the separator and adds the colors if enabled.
     */
    format(level: LoggerLevel, ...args: any[]): any[];
    private formatArgs;
    private formatPrimitives;
    /**
     * Gets the level from the environment variables.
     */
    protected static getLevelFromEnvironment(name: string): LoggerLevel | undefined;
    /**
     * Gets the status from the environment variables.
     */
    protected static getStatusFromEnvironment(name: string): LoggerStatus | undefined;
    /**
     * Checks if the logger is disabled.
     */
    get isDisabled(): boolean;
    /**
     * Checks if the logger is enabled.
     */
    get isEnabled(): boolean;
    /**
     * Checks if the verbose level is disabled.
     */
    protected get isLevelVerboseDisabled(): boolean;
    /**
     * Checks if the debug level is disabled.
     */
    protected get isLevelDebugDisabled(): boolean;
    /**
     * Checks if the info level is disabled.
     */
    protected get isLevelInfoDisabled(): boolean;
    /**
     * Checks if the warn level is disabled.
     */
    protected get isLevelWarnDisabled(): boolean;
}
