import { LocalizationInit, LocalizationPack, LocalizationVariables } from '../definitions/interfaces.js';
import { Storage } from '../definitions/types.js';
/**
 * The Localization class is used to localize anything that can be localized.
 *
 * - The language will persist in the storage, by default it will be stored in memory.
 * - The path of the localized string supports dot notation, for example: 'path.to.the.value'.
 * - The variables support dot notation as well and can be used inside the localized string, for example: 'Hello {name}!'.
 * - The instance also supports default variables, which can be overridden by the variables passed to the get method.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/localization)
 */
export declare class Localization<T extends LocalizationVariables = LocalizationVariables> {
    /**
     * The language that will be used to localize.
     */
    language: string;
    /**
     * The packs that will be used to localize.
     */
    packs: LocalizationPack[];
    /**
     * The storage that will be used to store the language.
     */
    storage: Storage;
    /**
     * The key that will be used to store the language.
     */
    storageKey: string;
    /**
     * The default variables that will be used to localize.
     */
    variables: T;
    constructor(init?: LocalizationInit<T>);
    /**
     * Retrieves the language from the storage and sets it.
     */
    initialize(): Promise<void | Error>;
    /**
     * Stores the language in the storage.
     */
    store(): Promise<void | Error>;
    /**
     * Adds the packs to the instance, if a pack for the language already exists, the data will be merged.
     */
    push(packs: LocalizationPack[]): void;
    push(...packs: LocalizationPack[]): void;
    /**
     * Retrieves the localized string from the pack.
     * Optionally you can pass variables that will be used to replace the variables inside the localized string.
     */
    get<T extends LocalizationVariables>(path: string, variables?: T): string;
    get<T extends LocalizationVariables>(language: string, path: string, variables?: T): string;
    /**
     * Sets the language.
     */
    setLanguage(language: string): void;
    /**
     * Sets the default variables.
     */
    setVariables(variables: T): void;
    /**
     * Checks if the pack has the path.
     */
    has(path: string): boolean;
    has(language: string, path: string): boolean;
    /**
     * Retrieves the pack by the language.
     */
    getPackByLanguage(language: string): LocalizationPack;
}
