import { RestApiConfig } from '../definitions/interfaces.js';
import { RequestMethod, WriteMode } from '../definitions/types.js';
import { FetchError } from './fetch-error.js';
import { FetchResponse } from './fetch-response.js';
import { Status } from './status.js';
/**
 * The RestAPI class manages the requests to a REST API.
 *
 * - The base URL of the API is automatically concatenated to the path of the requests.
 * - The config of the API is automatically merged with the config of the requests.
 * - The status of the requests is automatically tracked and can be accessed through the status property.
 * - The requests are sent with the Fetch class, so all features of the Fetch class are available.
 *
 * [Aracna Reference](https://aracna.dariosechi.it/core/classes/rest-api)
 */
export declare class RestAPI<T extends RestApiConfig = RestApiConfig, U = undefined> {
    /**
     * The base URL of the REST API.
     */
    protected baseURL: string;
    /**
     * The default config of the REST API.
     */
    protected config: T;
    /**
     * The status of the requests.
     */
    readonly status: Status;
    constructor(baseURL?: string, config?: T);
    /**
     * Sends a request to the REST API.
     */
    protected send<V, W, X = U>(method: RequestMethod, path: string, body?: W, config?: T): Promise<FetchResponse<V> | FetchError<X>>;
    /**
     * Sends a CONNECT request to the REST API.
     */
    connect<V, W = U>(path: string, config?: T): Promise<FetchResponse<V> | FetchError<W>>;
    /**
     * Sends a DELETE request to the REST API.
     */
    delete<V, W = U>(path: string, config?: T): Promise<FetchResponse<V> | FetchError<W>>;
    /**
     * Sends a GET request to the REST API.
     */
    get<V, W = U>(path: string, config?: T): Promise<FetchResponse<V> | FetchError<W>>;
    /**
     * Sends a HEAD request to the REST API.
     */
    head(path: string, config?: T): Promise<FetchResponse | FetchError>;
    /**
     * Sends an OPTIONS request to the REST API.
     */
    options<V, W = U>(path: string, config?: T): Promise<FetchResponse<V> | FetchError<W>>;
    /**
     * Sends a PATCH request to the REST API.
     */
    patch<V, W, X = U>(path: string, body?: W, config?: T): Promise<FetchResponse<V> | FetchError<X>>;
    /**
     * Sends a POST request to the REST API.
     */
    post<V, W, X = U>(path: string, body?: W, config?: T): Promise<FetchResponse<V> | FetchError<X>>;
    /**
     * Sends a PUT request to the REST API.
     */
    put<V, W, X = U>(path: string, body?: W, config?: T): Promise<FetchResponse<V> | FetchError<X>>;
    /**
     * Sends a TRACE request to the REST API.
     */
    trace(path: string, config?: T): Promise<FetchResponse | FetchError>;
    /**
     * Sends a POST request if the mode is 'create' or a PUT request if the mode is 'update' to the REST API.
     */
    write<V, W, X = U>(mode: WriteMode, path: string, body?: W, config?: T): Promise<FetchResponse<V> | FetchError<X>>;
    /**
     * Transforms the body of the request.
     */
    transformBody<V>(method: RequestMethod, path: string, body: V | undefined, config: T): Promise<any>;
    /**
     * Transforms the query parameters of the request.
     */
    transformQueryParameters<V>(method: RequestMethod, path: string, body: V | undefined, config: T): Promise<string>;
    /**
     * Handles the error of the request.
     */
    handleError<V, W = U>(method: RequestMethod, path: string, body: V | undefined, config: T, error: FetchError<W>): Promise<boolean>;
    /**
     * Handles the pending state of the request.
     */
    handlePending<V>(method: RequestMethod, path: string, body: V | undefined, config: T): Promise<boolean>;
    /**
     * Handles the success of the request.
     */
    handleSuccess<V, W>(method: RequestMethod, path: string, body: W | undefined, config: T, response: FetchResponse<V>): Promise<boolean>;
    /**
     * Sets the status of the request.
     */
    protected setCallStatus(method: RequestMethod, path: string, config: T, status: string): void;
    getBaseURL(): string;
    getConfig(): T;
    setBaseURL(baseURL: string): this;
    setConfig(config: T): this;
    /**
     * Checks if the config status is settable.
     */
    protected isConfigStatusSettable(config: RestApiConfig, status: string): boolean;
}
